---
name: json2video-pinterest
description: Generate Pinterest-optimized vertical videos using JSON2Video API. Supports AI-generated or URL-based images, AI-generated or provided voiceovers, optional subtitles, and zoom effects. Use when creating video content for Pinterest affiliate marketing, creating vertical social media videos, automating video production with JSON2Video API, or generating videos with voiceovers and subtitles.
---

# JSON2Video Pinterest Skill

Generate vertical videos (1080x1920) optimized for Pinterest using the JSON2Video API.

## Prerequisites

1. **JSON2Video API Key**: Sign up at https://json2video.com/get-api-key/
2. **Set Environment Variable**:
   ```bash
   export JSON2VIDEO_API_KEY="your_api_key_here"
   ```

## Quick Start

Create a video using a JSON configuration file:

```bash
python3 scripts/generate_video.py --config my-video.json --wait
```

## Configuration Format

The video is defined as an array of scenes. Each scene contains:

| Property | Type | Description |
|----------|------|-------------|
| `image` | object | Image configuration (AI-generated or URL) |
| `voice` | object | Voice configuration (generated TTS or URL) |
| `text_overlay` | string | Optional text displayed on scene |
| `subtitles` | boolean | Enable/disable subtitles |
| `zoom_effect` | boolean | Add Ken Burns zoom effect |
| `duration` | number | Override scene duration (seconds) |

### Image Configuration

**AI-Generated Image:**
```json
{
  "image": {
    "source": "ai",
    "ai_provider": "flux-schnell",
    "ai_prompt": "A minimalist workspace with laptop..."
  }
}
```

**Available AI Providers:**
- `flux-pro` - Highest quality, realistic images
- `flux-schnell` - Fast generation, good quality
- `freepik-classic` - Digital artwork style

**URL-Based Image:**
```json
{
  "image": {
    "source": "https://example.com/image.jpg"
  }
}
```

### Voice Configuration

**AI-Generated Voice (TTS):**
```json
{
  "voice": {
    "source": "generated",
    "text": "Your voiceover text here",
    "voice_id": "en-US-EmmaMultilingualNeural",
    "model": "azure"
  }
}
```

**Provided Audio File:**
```json
{
  "voice": {
    "source": "https://example.com/voiceover.mp3"
  }
}
```

**Note on Scene Duration**: The voiceover determines scene length automatically. Each scene's duration matches its audio length. For provided audio files, ensure they match your intended scene timing.

### Complete Example

```json
{
  "resolution": "instagram-story",
  "quality": "high",
  "cache": true,
  "scenes": [
    {
      "image": {
        "source": "ai",
        "ai_provider": "flux-schnell",
        "ai_prompt": "Affiliate marketing workspace with laptop and coffee"
      },
      "voice": {
        "source": "generated",
        "text": "Here's how to make money with affiliate marketing",
        "voice_id": "en-US-Neural2-F"
      },
      "text_overlay": "Affiliate Marketing 101",
      "subtitles": true,
      "zoom_effect": true
    }
  ]
}
```

## Advanced: Split Long Voiceover into Scenes

For long scripts, split into multiple scenes with shorter voice segments:

```json
{
  "scenes": [
    {
      "image": { "source": "ai", "ai_prompt": "Hook image" },
      "voice": { "source": "generated", "text": "Attention-grabbing hook..." },
      "zoom_effect": true
    },
    {
      "image": { "source": "ai", "ai_prompt": "Step 1 image" },
      "voice": { "source": "generated", "text": "Step one is to..." },
      "zoom_effect": true
    },
    {
      "image": { "source": "ai", "ai_prompt": "CTA image" },
      "voice": { "source": "generated", "text": "Click the link in bio..." },
      "zoom_effect": false
    }
  ]
}
```

## Command Reference

**Create video from config:**
```bash
python3 scripts/generate_video.py --config video.json --wait
```

**Create without waiting:**
```bash
python3 scripts/generate_video.py --config video.json --no-wait
```

**Check status of existing project:**
```bash
python3 scripts/generate_video.py --project-id YOUR_PROJECT_ID
```

## Resolution Options

| Resolution | Dimensions | Use Case |
|------------|------------|----------|
| `instagram-story` | 1080x1920 | **Pinterest/Reels/Stories** (recommended) |
| `instagram-feed` | 1080x1080 | Square posts |
| `full-hd` | 1920x1080 | Landscape YouTube |
| `hd` | 1280x720 | Standard HD |
| `custom` | Any | Custom dimensions |

## Voice Models & IDs

### Azure (Default - FREE, no credits consumed)

**Voice format:** `en-US-EmmaMultilingualNeural`

Common Azure voices:
- `en-US-EmmaMultilingualNeural` - Female, natural (recommended)
- `en-US-GuyNeural` - Male, professional
- `en-US-JennyNeural` - Female, friendly
- `en-GB-SoniaNeural` - British female
- `en-GB-RyanNeural` - British male

See [Microsoft Azure Speech Voices](https://learn.microsoft.com/en-us/azure/ai-services/speech-service/language-support?tabs=tts) for full list.

### ElevenLabs (Premium - consumes credits)

**Voice names:** Natural names like `Bella`, `Antoni`, `Nova`, `Shimmer`

Available voices: Daniel, Serena, Antoni, Bella, Nova, Shimmer, and more.

See [ElevenLabs Voice Library](https://elevenlabs.io/app/voice-library) for full list.

## Security

- **API Key is NEVER stored in skill files**
- API key must be set as environment variable `JSON2VIDEO_API_KEY`
- Script validates key exists before any API calls

## Example Files

- `scripts/example-config.json` - Basic example with one scene
- `scripts/example-advanced.json` - Multi-scene affiliate marketing video

## Advanced Usage

See [ADVANCED.md](ADVANCED.md) for:
- Multi-scene video architecture patterns
- Image source strategies (AI vs URL vs hybrid)
- Voiceover patterns and best practices
- Subtitle styling options
- Pinterest-specific content tips
- Batch processing workflows
- Credit consumption optimization

## Troubleshooting

**Error: "JSON2VIDEO_API_KEY environment variable not set"**
→ Run: `export JSON2VIDEO_API_KEY="your_key"`

**Error: "Render failed"**
→ Check: Image URLs are publicly accessible
→ Check: AI prompts don't violate content policies
→ Check: Audio files are valid MP3/WAV

**Video takes too long**
→ Enable `cache: true` in config
→ Use `flux-schnell` instead of `flux-pro` for faster generation
→ Pre-generate AI images and use URLs instead

## Python API Usage

For programmatic use in other scripts:

```python
from scripts.generate_video import create_pinterest_video

scenes = [
    {
        "image": {"source": "ai", "ai_prompt": "..."},
        "voice": {"source": "generated", "text": "..."},
        "subtitles": True,
        "zoom_effect": True
    }
]

video_url = create_pinterest_video(scenes, wait=True)
print(f"Video ready: {video_url}")
```
