#!/bin/bash
# autonomous-tasks 健康检查脚本
# 用于检测系统状态并在可能的情况下自动修复

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(dirname "$SCRIPT_DIR")"

echo "=== Autonomous-Tasks 健康检查 ==="
echo "检查时间: $(date '+%Y-%m-%d %H:%M:%S')"
echo ""

# 状态文件
ERRORS=0
WARNINGS=0

# 1. 检查核心文件
echo "📁 检查核心文件..."
CORE_FILES=(
    "$SKILL_DIR/SKILL.md"
    "$SKILL_DIR/AUTONOMOUS.md"
    "$SKILL_DIR/_meta.json"
    "$SKILL_DIR/memory/tasks-log.md"
    "$SKILL_DIR/memory/backlog.md"
)

for file in "${CORE_FILES[@]}"; do
    if [ -f "$file" ]; then
        echo "  ✅ $(basename $file)"
    else
        echo "  ❌ $(basename $file) 不存在"
        ((ERRORS++))
    fi
done

# 2. 检查目录结构
echo ""
echo "📂 检查目录结构..."
DIRS=("research" "drafts" "apps" "scripts" "memory")
for dir in "${DIRS[@]}"; do
    if [ -d "$SKILL_DIR/$dir" ]; then
        echo "  ✅ $dir/"
    else
        echo "  ⚠️  $dir/ 不存在，创建中..."
        mkdir -p "$SKILL_DIR/$dir"
        echo "  ✅ $dir/ 已创建"
        ((WARNINGS++))
    fi
done

# 3. 检查任务日志格式
echo ""
echo "📋 检查任务日志..."
TASK_LOG="$SKILL_DIR/memory/tasks-log.md"
if [ -f "$TASK_LOG" ]; then
    # 检查是否有损坏的记录（没有完成标记）
    CORRUPT=$(grep -c "^- \[ \]" "$TASK_LOG" 2>/dev/null || echo "0")
    if [ "$CORRUPT" -gt 0 ]; then
        echo "  ⚠️  发现 $CORRUPT 个未完成的任务记录"
        ((WARNINGS++))
    else
        # 统计已完成任务
        DONE_COUNT=$(grep -c "✅ TASK-" "$TASK_LOG" 2>/dev/null || echo "0")
        echo "  ✅ 任务日志正常 (已完成 $DONE_COUNT 个任务)"
    fi
else
    echo "  ❌ 任务日志不存在"
    ((ERRORS++))
fi

# 4. 检查 cron 配置
echo ""
echo "⏰ 检查 Cron 配置..."
CRON_NAME="autonomous-tasks"
if command -v openclaw &> /dev/null; then
    CRON_STATUS=$(openclaw cron list 2>/dev/null | grep "$CRON_NAME" || echo "")
    if [ -n "$CRON_STATUS" ]; then
        echo "  ✅ Cron 任务已配置: $CRON_NAME"
    else
        echo "  ⚠️  Cron 任务未找到: $CRON_NAME"
        ((WARNINGS++))
    fi
else
    echo "  ⚠️  openclaw 命令不可用"
    ((WARNINGS++))
fi

# 5. 检查最近执行记录
echo ""
echo "🕐 检查最近执行..."
LAST_TASK=$(tail -5 "$TASK_LOG" 2>/dev/null | grep "TASK-" | tail -1)
if [ -n "$LAST_TASK" ]; then
    echo "  ✅ 最近任务: $LAST_TASK"
else
    echo "  ⚠️  无法获取最近任务"
    ((WARNINGS++))
fi

# 总结
echo ""
echo "=== 健康检查总结 ==="
if [ $ERRORS -eq 0 ] && [ $WARNINGS -eq 0 ]; then
    echo "🎉 系统状态: 优秀"
    exit 0
elif [ $ERRORS -eq 0 ]; then
    echo "✅ 系统状态: 正常 (警告: $WARNINGS 项)"
    exit 0
else
    echo "❌ 系统状态: 需要关注 (错误: $ERRORS, 警告: $WARNINGS)"
    exit 1
fi
