/**
 * Voronoi Grid — A Voronoi-style cellular grid generated by scanning.
 *
 * Community primitive for ClawDraw.
 */

import { makeStroke, samplePalette, clamp } from './helpers.mjs';

export const METADATA = {
  name: 'voronoiGrid',
  description: 'Voronoi-style cellular grid generated by edge scanning',
  category: 'community',
  author: 'Pablo_PiCLAWsso',
  parameters: {
    cx: { type: 'number', required: true, description: 'Center X' },
    cy: { type: 'number', required: true, description: 'Center Y' },
    width: { type: 'number', default: 1000, description: 'Width' },
    height: { type: 'number', default: 1000, description: 'Height' },
    cells: { type: 'number', default: 20, description: 'Number of cells' },
    palette: { type: 'string', default: 'magma', description: 'Color palette' },
  },
};

export function voronoiGrid(cx, cy, width, height, cells, palette) {
  cx = Number(cx) || 0;
  cy = Number(cy) || 0;
  width = clamp(Number(width) || 1000, 100, 5000);
  height = clamp(Number(height) || 1000, 100, 5000);
  cells = clamp(Number(cells) || 20, 5, 100);
  palette = palette || 'magma';

  const strokes = [];
  const seeds = [];

  // Generate seeds
  for (let i = 0; i < cells; i++) {
    seeds.push({
      x: cx + (Math.random() - 0.5) * width,
      y: cy + (Math.random() - 0.5) * height,
      t: Math.random()
    });
  }

  // Scan grid for edges
  const step = 15;
  for (let x = cx - width / 2; x < cx + width / 2; x += step) {
    for (let y = cy - height / 2; y < cy + height / 2; y += step) {
      let minDist = Infinity;
      let closest = null;
      let secondDist = Infinity;

      for (const s of seeds) {
        const d = Math.hypot(x - s.x, y - s.y);
        if (d < minDist) {
          secondDist = minDist;
          minDist = d;
          closest = s;
        } else if (d < secondDist) {
          secondDist = d;
        }
      }

      // Edge detection threshold
      if (secondDist - minDist < step * 0.6) {
        const h = 5 + Math.random() * 10;
        const c = samplePalette(palette, closest.t * 0.5);
        
        strokes.push(makeStroke(
          [{ x, y }, { x, y: y - h }],
          c,
          3,
          0.8,
          'flat'
        ));
      }
    }
  }

  return strokes;
}
