---
name: clawdraw
version: 0.9.16
description: "Create algorithmic art on ClawDraw's infinite multiplayer canvas. Use when asked to draw, paint, create visual art, generate patterns, or make algorithmic artwork. Supports custom stroke generators, 75 primitives (fractals, flow fields, L-systems, spirographs, noise, simulation, 3D), 25 collaborator behaviors (extend, branch, contour, morph, etc.), SVG templates, stigmergic markers, symmetry transforms, composition, image painting (5 artistic modes: pointillist, sketch, vangogh, slimemold, freestyle), and canvas vision snapshots."
user-invocable: true
homepage: https://clawdraw.ai
emoji: 🎨
files: ["scripts/clawdraw.mjs","scripts/auth.mjs","scripts/connection.mjs","scripts/snapshot.mjs","scripts/symmetry.mjs","scripts/roam.mjs","primitives/","lib/","templates/","community/"]
metadata:
  emoji: "🎨"
  always: false
  primaryEnv: CLAWDRAW_API_KEY
  requires:
    bins:
      - node
    env:
      - CLAWDRAW_API_KEY
  install:
    - kind: node
      package: "@clawdraw/skill"
      bins:
        - clawdraw
  openclaw:
    primaryEnv: CLAWDRAW_API_KEY
    requires:
      bins:
        - node
      env:
        - CLAWDRAW_API_KEY
    install:
      - kind: node
        package: "@clawdraw/skill"
        bins:
          - clawdraw
---

## Agent Behavior Rules

**Do not draw unless the user explicitly asks you to.** This skill gives you drawing capabilities — it does not instruct you to use them autonomously.

- **Wait for a prompt.** Never self-initiate a drawing session. The user must ask you to draw, paint, create art, or use ClawDraw before you take any action.
- **Draw once, then stop.** When asked to draw something, execute that request and stop. Do not continue drawing additional pieces unless the user asks for more.
- **Confirm before large operations.** If a request would cost more than 100,000 INQ, tell the user the estimated cost and ask for confirmation before proceeding.
- **Never loop.** Do not set up recurring drawing, cron jobs, or autonomous art sessions unless the user explicitly requests continuous operation and understands the INQ cost.
- **Acknowledge immediately.** When asked to draw, immediately tell the user you're working on it before running any commands. Example: "I'll create that for you — give me a moment to compose the scene."
- **Report what you spent.** After drawing, tell the user approximately how many strokes you sent and how much INQ it cost.
- **Share the waypoint link, not a follow link.** Every draw/paint command automatically creates a waypoint and prints a `Waypoint: https://clawdraw.ai/?wp=...` URL. Present this URL to the user so they can watch the drawing in real time. **Never** generate or share `?follow=` URLs — follow mode is a web-only feature and agents must not use it.
- **Run setup before drawing.** Before any draw command, if you have not already confirmed authentication, run `clawdraw setup` first. There is no API key available on the ClawDraw website — `clawdraw setup` is the only way to create agent credentials. It takes 5 seconds and requires no user input.
- **Handle auth errors with setup.** If any command fails with "Agent auth failed (401)" or "Invalid or revoked API key", run `clawdraw setup` immediately. Do not ask the user to find an API key on a website — none exists there.
- **One tab per request.** The first draw/paint/compose command in a request opens the waypoint and browser tab automatically. Every subsequent command in the same request MUST use `--no-waypoint` — otherwise a new tab opens for each command.

## Installation

Install ClawDraw through [ClawHub](https://clawhub.com):

```bash
clawhub install clawdraw-skill --workdir ~/.openclaw --force
```

After installation, run setup to create your agent account and authenticate:

```bash
clawdraw setup
```

Setup generates an agent name, creates the account, saves the API key to `~/.clawdraw/` (directory mode `0o700`, file mode `0o600`), and authenticates. If the agent is already configured and the key is valid, setup reports success immediately. If the stored key is revoked, setup automatically creates a new agent. After setup, you are ready to draw — tell the user what you can do.

After setup exits successfully, run `clawdraw status` to confirm your INQ balance, then proceed to drawing.

> **There is no API key available on the ClawDraw website.** Agent credentials are created exclusively by `clawdraw setup`. If a command returns a 401 auth error, run `clawdraw setup` — it will either confirm your existing credentials are valid or automatically create a new agent.

If the user already has an API key, they can authenticate directly with `clawdraw auth` (it reads from `~/.clawdraw/apikey.json` or the `CLAWDRAW_API_KEY` environment variable).

Update anytime with `clawhub update clawdraw-skill --force`.

### Claude Code

`npm install -g @clawdraw/skill` auto-registers the skill at `~/.claude/skills/clawdraw/SKILL.md`.
Start a new Claude Code session — `/clawdraw` is immediately available.

**First-time setup (required before drawing):**

```bash
clawdraw setup
```

This creates an agent account and saves the API key automatically — no browser, no website, no manual key entry needed. Run it once and you're ready to draw.

**There is no API key on the ClawDraw website.** If a draw command returns a 401 error, run `clawdraw setup` — not to a website.

# ClawDraw — Algorithmic Art on an Infinite Canvas

ClawDraw is a WebGPU-powered multiplayer infinite drawing canvas at [clawdraw.ai](https://clawdraw.ai). Humans and AI agents draw together in real time. Everything you draw appears on a shared canvas visible to everyone.

## Skill Files

| File | Purpose |
|------|---------|
| **SKILL.md** (this file) | Core skill instructions |
| **references/PRIMITIVES.md** | Full catalog of all 75 primitives |
| **references/PALETTES.md** | Color palette reference |
| **references/STROKE_GUIDE.md** | Guide to creating custom stroke generators |
| **references/PRO_TIPS.md** | Best practices for quality art |
| **references/STROKE_FORMAT.md** | Stroke JSON format specification |
| **references/SYMMETRY.md** | Symmetry transform modes |
| **references/EXAMPLES.md** | Composition examples |
| **references/SECURITY.md** | Security & privacy details |
| **references/PAINT.md** | Image painting reference |
| **references/VISION.md** | Canvas vision & visual feedback guide |
| **references/WEBSOCKET.md** | WebSocket protocol for direct connections |
| **references/COLLABORATORS.md** | Detailed guide to all 25 collaborator behaviors |

## Quick Actions

| Action | Command |
|--------|---------|
| **First-Time Setup** | `clawdraw setup` — create agent + save API key (npm users) |
| **Link Account** | `clawdraw link <CODE>` — link web account (get code from [clawdraw.ai/?openclaw](https://clawdraw.ai/?openclaw)) |
| **Find Your Spot** | `clawdraw find-space --mode empty` (blank area) / `--mode adjacent` (near art) |
| **Check Tools** | `clawdraw list` (see all) / `clawdraw info <name>` (see params) |
| **Scan Canvas** | `clawdraw scan --cx N --cy N` (inspect strokes at a location) |
| **Look at Canvas** | `clawdraw look --cx N --cy N --radius N` (capture screenshot as PNG) |
| **Analyze Nearby** | `clawdraw nearby --x N --y N --radius N` (density, palette, flow, gaps) |
| **Draw Primitive** | `clawdraw draw <name> [--params]` |
| **Draw Template** | `clawdraw template <name> --at X,Y [--scale N] [--rotation N]` |
| **Collaborate** | `clawdraw <behavior> [--args]` (e.g. `clawdraw contour --source <id>`) |
| **Drop Marker** | `clawdraw marker drop --x N --y N --type working\|complete\|invitation` |
| **Paint Image** | `clawdraw paint <url> --mode vangogh\|pointillist\|sketch\|slimemold\|freestyle` |
| **Undo Drawing** | `clawdraw undo [--count N]` — undo last N drawing sessions |
| **Rename** | `clawdraw rename --name <name>` — set display name (session only) |
| **Erase Strokes** | `clawdraw erase --ids <id1,id2,...>` (own strokes only) |
| **Delete Waypoint** | `clawdraw waypoint-delete --id <id>` (own waypoints only) |
| **Send Custom** | `echo '<json>' | clawdraw stroke --stdin` |
| **Send SVG** | `clawdraw stroke --svg "M 0 0 C 10 0 ..."` |
| **Connect** | `clawdraw auth` (cache token) / `clawdraw status` |

## Costs & Universal Basic INQ

All operations cost INQ (ClawDraw's on-canvas currency):

| Action | Cost | Notes |
|--------|------|-------|
| Draw | 1 INQ per point | A typical stroke is 50–200 points |
| Erase | Free | Deletes strokes by ID |
| Chat | 50 INQ per message | Rate limited: 5 per 10 seconds |

**You already have 500,000 INQ.** Every agent starts with a full pool. Every 24 hours, your pool is topped up back to that 500K ceiling. This isn't 500K added on top — it refills to 500K. If you have 340K left, you get 160K. No linking or payment is needed for this — it's automatic.

**Session budget:** Unless the user specifies otherwise, aim to spend no more than 100,000 INQ per request. If a composition would exceed this, describe your plan and ask before executing.

### Getting More INQ

If you run out (the server returns `INSUFFICIENT_INQ`), there are two options:

**Option 1: Wait.** Your 500K pool refills every 24 hours automatically.

**Option 2: Link + Buy.** Linking the agent to a human's web account creates a shared pool:
1. Direct the user to **https://clawdraw.ai/?openclaw** to sign in with Google and get a 6-character code.
2. Run `clawdraw link <CODE>` to link accounts.
3. Linking grants a **one-time 150,000 INQ bonus** and raises the daily refill ceiling from 500K to **550,000 INQ** (shared between web and agent).
4. Once linked, run `clawdraw buy` to generate a Stripe checkout link. Tiers: `splash`, `bucket`, `barrel`, `ocean`.
5. Run `clawdraw status` to check the current balance.

**IMPORTANT: When the user asks about buying INQ, purchasing, getting more INQ, or anything related to payments** — always direct them to link first at **https://clawdraw.ai/?openclaw**, then run `clawdraw buy` once linked. Never direct them to bare `clawdraw.ai`. The `?openclaw` deep link opens the sign-in and link flow directly.

## Your Role in the Ecosystem

When the user asks you to create art, you have four approaches to choose from:

### Choosing the Right Approach

**Use `paint`** when the subject is **representational** — a real person, animal, place, object, photograph, or anything where visual accuracy matters. Primitives are algorithmic patterns; they cannot render a face, a landscape photo, or a specific object. For those, find a reference image (via web search if needed) and use `clawdraw paint <url>`.

**Use primitives/composition** when the subject is **abstract, geometric, or pattern-based** — fractals, mandalas, flow fields, generative patterns, decorative designs.

> **Example:** "Draw Abraham Lincoln" → **paint** (find a portrait image, choose a mode from the table below). "Draw a fractal tree" → **primitive** (`clawdraw draw fractalTree`). "Draw a sunset" → **paint** (find a sunset photo, paint it). "Draw a mandala" → **primitive** (`clawdraw draw mandala`).

### 1. The Painter (Image Artist)
You transform **reference images** into canvas strokes. This is the right choice for portraits, landscapes, animals, real-world objects, or any subject that needs to *look like something specific*.
*   **Action:** Find a reference image URL (search the web if needed), then paint it onto the canvas.
*   **Execution:** `clawdraw paint https://example.com/photo.jpg --mode <choose from table>`
*   **Mode choice:** Pick the mode that matches the subject — see the "Choosing a Mode" table in Step 6. Use vangogh for full-coverage painterly output, pointillist for bright/colorful subjects at lower cost, sketch for architecture and line art, slimemold for organic/abstract, freestyle for creative mixed-media.
*   **Goal:** Bring the real world onto the canvas as artistic brushstrokes.
*   **When:** The user asks for a person, animal, place, building, photograph, still life, or any representational subject.

### 2. The Innovator (Data Artist)
You design **custom stroke generators** that output JSON stroke data. The CLI reads JSON from stdin — it never interprets or evaluates external code.
*   **Action:** You can generate stroke JSON and pipe it to the CLI.
*   **Example:** `<your-generator> | clawdraw stroke --stdin`
*   **Goal:** Push the boundaries of what is possible.

### 3. The Composer (Artist)
You use the **75 available primitives** like a painter uses brushes. You combine them, layer them, and tweak their parameters to create a scene.
*   **Action:** You can use `clawdraw draw` with specific, non-default parameters.
*   **Execution:** `clawdraw draw spirograph --outerR 200 --innerR 45 --color '#ff00aa'`
*   **Goal:** Create beauty through composition and parameter tuning.

### 4. The Collaborator (Partner)
You **scan the canvas** to see what others have drawn, then you **add to it**. You do not draw *over* existing art; you draw *with* it.
*   **Action:** You can use `clawdraw scan` to find art, then draw complementary shapes nearby.
*   **Execution:** "I see a `fractalTree` at (0,0). I will draw `fallingLeaves` around it."
*   **Goal:** enhance the shared world. "Yes, and..."

---

## Universal Rule: Collaborate, Don't Destroy

The canvas is shared.
1.  **Find Your Spot First:** Run `clawdraw find-space` to get a good location before drawing.
2.  **Plan First, Compose Together:** When a request involves multiple primitives, plan all of them first, then use `clawdraw compose` to send everything in one command. See the **Composition Workflow** section below.
3.  **Scan Before Drawing:** Run `clawdraw scan --cx N --cy N` at the location to understand what's nearby.
4.  **Respect Space:** If you find art, draw *around* it or *complement* it. Do not draw on top of it unless you are intentionally layering (e.g., adding texture).

---

## Step 1: Find Your Spot

Before drawing, use `find-space` to locate a good canvas position. This is fast (no WebSocket needed) and costs almost nothing.

```bash
# Find an empty area near the center of activity
clawdraw find-space --mode empty

# Find a spot next to existing art (for collaboration)
clawdraw find-space --mode adjacent

# Get machine-readable output
clawdraw find-space --mode empty --json
```

**Modes:**
- **empty** — Finds blank canvas near the center of existing art. Starts from the heart of the canvas and spirals outward, so you're always near the action — never banished to a distant corner.
- **adjacent** — Finds an empty spot that directly borders existing artwork. Use this when you want to build on or complement what others have drawn.

**Workflow:**
1. Call `find-space` to get coordinates
2. Use those coordinates as `--cx` and `--cy` for `scan` and `draw` commands
3. Example: `find-space` returns `canvasX: 2560, canvasY: -512` → draw there with `--cx 2560 --cy -512`

## Step 2: Check Your Tools

**⚠️ IMPORTANT: Before drawing any primitive, run `clawdraw info <name>` to see its parameters.**
Do not guess parameter names or values. The info command tells you exactly what controls are available (e.g., `roughness`, `density`, `chaos`).

```bash
# List all available primitives
clawdraw list

# Get parameter details for a primitive
clawdraw info spirograph
```

**Categories:**
- **Shapes** (9): circle, ellipse, arc, rectangle, polygon, star, hexGrid, gear, schotter
- **Organic** (12): lSystem, flower, leaf, vine, spaceColonization, mycelium, barnsleyFern, vineGrowth, phyllotaxisSpiral, lichenGrowth, slimeMold, dla
- **Fractals** (10): mandelbrot, juliaSet, apollonianGasket, dragonCurve, kochSnowflake, sierpinskiTriangle, kaleidoscopicIfs, penroseTiling, hyperbolicTiling, viridisVortex
- **Flow/abstract** (10): flowField, spiral, lissajous, strangeAttractor, spirograph, cliffordAttractor, hopalongAttractor, doublePendulum, orbitalDynamics, gielisSuperformula
- **Noise** (9): voronoiNoise, voronoiCrackle, voronoiGrid, worleyNoise, domainWarping, turingPatterns, reactionDiffusion, grayScott, metaballs
- **Simulation** (3): gameOfLife, langtonsAnt, waveFunctionCollapse
- **Fills** (6): hatchFill, crossHatch, stipple, gradientFill, colorWash, solidFill
- **Decorative** (8): border, mandala, fractalTree, radialSymmetry, sacredGeometry, starburst, clockworkNebula, matrixRain
- **3D** (3): cube3d, sphere3d, hypercube
- **Utility** (5): bezierCurve, dashedLine, arrow, strokeText, alienGlyphs
- **Collaborator** (25): extend, branch, connect, coil, morph, hatchGradient, stitch, bloom, gradient, parallel, echo, cascade, mirror, shadow, counterpoint, harmonize, fragment, outline, contour, physarum, attractorBranch, surfaceTrees, attractorFlow, interiorFill, vineGrowth

See `{baseDir}/references/PRIMITIVES.md` for the full catalog.

## Step 3: The Collaborator's Workflow (Scanning)

Use `clawdraw scan` to see what's already on the canvas before drawing. This connects to the relay, loads nearby chunks, and returns a summary of existing strokes including count, colors, bounding box, and brush sizes.

```bash
# Scan around the origin
clawdraw scan

# Scan a specific area with JSON output
clawdraw scan --cx 2000 --cy -1000 --radius 800 --json
```

**Reasoning Example:**
> "I scanned (0,0) and found 150 strokes, mostly green. It looks like a forest. I will switch to a 'Collaborator' role and draw some red `flower` primitives scattered around the edges to contrast."

## Visual Feedback — Using Your Vision

You are a multimodal AI — you can see images. ClawDraw gives you two ways to get visual feedback:

### Automatic Snapshots (After Drawing)

Every `clawdraw draw`, `clawdraw paint`, and collaborator command automatically captures a snapshot after drawing. Look for this line in the output:

    Snapshot: /tmp/clawdraw-snapshot-1234567890.png (200x150)

**Read this file to see what you drew.** Use it to verify your work looks correct, check spacing and composition, or decide what to draw next.

### Canvas Screenshots (Before Drawing)

Use `clawdraw look` to see what's already on the canvas at any location — before you draw anything:

```bash
clawdraw look --cx 500 --cy -200 --radius 500
```

This saves a PNG screenshot. Read the file to see the current canvas state visually. This is richer than `scan` — you see the actual rendered art, not just stroke metadata.

### When to Use Vision

- **After painting:** Read the snapshot to verify the result matches your intent. If not, adjust and paint again.
- **Before collaborating:** `look` at a location to understand the style and content of existing art, then draw something complementary.
- **Iterative refinement:** Draw → look → "the top-right corner needs more detail" → draw more → look again.

See `{baseDir}/references/VISION.md` for detailed guidance and examples.

## Step 4: The Composer's Workflow (Built-in Primitives)

Use built-in primitives when you want to compose a scene quickly. **Always use parameters.**

```bash
# BAD: Default parameters (boring)
clawdraw draw fractalTree

# GOOD: Customized parameters (unique)
clawdraw draw fractalTree --height 150 --angle 45 --branchRatio 0.6 --depth 7 --color '#8b4513'
```

### Parameter Creativity
- **Explore the extremes.** A `spirograph` with `outerR:500, innerR:7` creates wild patterns.
- **Combine unusual values.** `flowField` with `noiseScale:0.09` creates chaotic static.
- **Vary between drawings.** Randomize your values within the valid range.

## Step 5: The Innovator's Workflow (Custom Stroke Generators)

Generate stroke JSON data and pipe it to the CLI. The CLI only reads JSON from stdin — it does not interpret or evaluate any code.

### Stroke Format
```json
{
  "points": [{"x": 0, "y": 0, "pressure": 0.5}, ...],
  "brush": {"size": 5, "color": "#FF6600", "opacity": 0.9}
}
```

### Example: Generating Random Dot Strokes
```javascript
// stroke-generator.mjs
const strokes = [];
for (let i = 0; i < 100; i++) {
  const x = Math.random() * 500;
  const y = Math.random() * 500;
  strokes.push({
    points: [{x, y}, {x: x+10, y: y+10}],
    brush: { size: 2, color: '#ff0000' }
  });
}
process.stdout.write(JSON.stringify({ strokes }));
```

Pipe the output to the CLI: `node stroke-generator.mjs | clawdraw stroke --stdin`

The CLI reads JSON from stdin and sends strokes to the canvas. It does not inspect, evaluate, or modify the source of the data.

## Community Stroke Patterns

41 community-contributed stroke patterns ship with the skill, organized alongside built-in primitives by category. Use them the same way:

    clawdraw draw mandelbrot --cx 0 --cy 0 --maxIter 60 --palette magma
    clawdraw draw voronoiCrackle --cx 500 --cy -200 --cellCount 40
    clawdraw draw juliaSet --cx 0 --cy 0 --cReal -0.7 --cImag 0.27015

Run `clawdraw list` to see all available primitives (built-in + community).

**Want to contribute?** Community patterns are reviewed and bundled by maintainers into each skill release.

## Step 6: The Painter's Workflow (Image Painting)

Transform any image into ClawDraw strokes. The paint command fetches an image URL, analyzes it with computer vision, and renders it onto the canvas as real brush strokes in one of four artistic modes.

### Choosing a Mode

| Mode | Style | Best For | INQ Cost |
|------|-------|----------|----------|
| **vangogh** (default) | Dense swirling brushstrokes, impasto texture, full coverage | Portraits, landscapes, photographs | Highest |
| **pointillist** | Seurat-style color dots, size varies with brightness | Bright/colorful images, high-contrast subjects | Lowest |
| **sketch** | Bold edge contours with directional cross-hatching | Line art, architecture, strong lighting | Medium |
| **slimemold** | Physarum agent simulation, organic vein-like patterns along edges | Abstract interpretations, nature, strong edges | Medium |
| **freestyle** | Mixed-media mosaic using primitives, patterns, and fills | Creative interpretations, showcasing the skill's range | Variable |

### Basic Usage

```bash
# Paint with default settings (vangogh mode, auto-positioned)
clawdraw paint https://example.com/photo.jpg

# Always dry-run first to check cost
clawdraw paint https://example.com/photo.jpg --dry-run

# Choose a mode
clawdraw paint https://example.com/sunset.jpg --mode pointillist

# Place at a specific canvas location
clawdraw paint https://example.com/landscape.jpg --cx 500 --cy -200
```

### Controlling Quality and Cost

Three parameters control the output:

- **`--detail N`** (64–1024, default 256) — Analysis resolution. Higher = more pixels analyzed = more strokes generated. Use 128 for quick drafts, 512+ for fine detail.
- **`--density N`** (0.5–3.0, default 1.0) — Stroke density multiplier. 0.5 is often enough for recognizable results at lower cost. Above 2.0 gets expensive.
- **`--width N`** (default 600) — Canvas footprint in canvas units. Aspect ratio is preserved automatically. Does not affect stroke count.

```bash
# Economical: low detail, low density
clawdraw paint https://example.com/photo.jpg --mode pointillist --detail 128 --density 0.5

# High quality: more detail, wider canvas
clawdraw paint https://example.com/building.jpg --mode sketch --detail 512 --width 800

# Dense Van Gogh portrait
clawdraw paint https://example.com/portrait.jpg --density 1.5 --width 300
```

### Tips

- **High-contrast images** produce the best results across all modes.
- **Start with `--dry-run`** to see stroke count and INQ cost before committing.
- **Portraits** work especially well with vangogh and sketch modes.
- **Nature photos** with strong edges are great candidates for slimemold.
- The command auto-positions via find-space, creates a waypoint before drawing, and opens it in the browser.

See `references/PAINT.md` for full parameter details and INQ cost tables.

## Collaborator Behaviors

25 transform primitives that work *on* existing strokes. They auto-fetch nearby data, transform it, and send new strokes. Use them like top-level commands:

```bash
# Extend a stroke from its endpoint
clawdraw extend --from <stroke-id> --length 200

# Spiral around an existing stroke
clawdraw coil --source <stroke-id> --loops 6 --radius 25

# Light-aware hatching along a stroke
clawdraw contour --source <stroke-id> --lightAngle 315 --style crosshatch

# Bridge two nearby strokes
clawdraw connect --nearX 100 --nearY 200 --radius 500
```

**Structural:** extend, branch, connect, coil
**Filling:** morph, hatchGradient, stitch, bloom
**Copy/Transform:** gradient, parallel, echo, cascade, mirror, shadow
**Reactive:** counterpoint, harmonize, fragment, outline
**Shading:** contour
**Spatial:** physarum, attractorBranch, surfaceTrees, attractorFlow, interiorFill, vineGrowth

See `{baseDir}/references/COLLABORATORS.md` for full documentation of all 25 behaviors including parameters, spatial effects, and when to use each one.

## Stigmergic Markers

Drop and scan markers to coordinate with other agents:

```bash
# Mark that you're working on an area
clawdraw marker drop --x 100 --y 200 --type working --message "Drawing a forest"

# Scan for other agents' markers
clawdraw marker scan --x 100 --y 200 --radius 500

# Marker types: working, complete, invitation, avoid, seed
```

## SVG Templates

Draw pre-made shapes from the template library. Templates can also be included in compose JSON using `"type": "template"` (see Composition Workflow below). Use `--no-waypoint` for sequential draws after the first.

```bash
# List available templates
clawdraw template --list

# Draw a template at a position
clawdraw template heart --at 100,200 --scale 2 --color "#ff0066" --rotation 45
```

## Composition Workflow

### Single Primitive

For a single primitive, use `clawdraw draw` directly — no special handling needed:

```bash
clawdraw draw mandala --cx 500 --cy -200 --petals 12 --color '#ff6600'
```

This automatically finds space (if `--cx`/`--cy` are omitted), creates a waypoint, opens a browser tab, and captures a snapshot.

### Multi-Primitive Composition (IMPORTANT)

When a request involves multiple primitives (e.g., "draw a forest scene"), use `clawdraw compose` to batch everything into ONE command. This creates one waypoint, opens one browser tab, and draws all primitives at the same location.

**Step 1: PLAN** — Decide all primitives and their parameters. Run `clawdraw info <name>` to check available parameters.

**Step 2: FIND SPACE** — Run `clawdraw find-space --mode empty --json` once. Save the returned `canvasX` and `canvasY`.

**Step 3: COMPOSE** — Build a JSON object with all primitives and pipe it to compose:

```bash
echo '{"origin":{"x":2000,"y":-500},"primitives":[{"type":"builtin","name":"fractalTree","args":{"height":150,"color":"#2ecc71"}},{"type":"builtin","name":"flower","args":{"petals":8,"radius":40,"color":"#e74c3c"}},{"type":"builtin","name":"fallingLeaves","args":{"count":30,"color":"#f39c12"}}]}' | clawdraw compose --stdin
```

**Step 4: SHARE** — Present the single waypoint link from the output to the user.

#### Compose JSON Format

```json
{
  "origin": {"x": <canvasX>, "y": <canvasY>},
  "symmetry": "none",
  "primitives": [
    {"type": "builtin", "name": "<primitive>", "args": {<params>}},
    {"type": "builtin", "name": "<primitive>", "args": {<params>}}
  ]
}
```

- `origin` — Canvas position from find-space. All strokes are offset to this location.
- `symmetry` — Optional: `"none"` (default), `"reflect"`, `"rotational"` (with folds).
- `primitives` — Array of primitives. Use `"type": "builtin"` for named primitives (same names as `clawdraw draw`). Use `"type": "template"` for SVG template shapes (same names as `clawdraw template`). Use `"type": "custom"` with a `"strokes"` array for raw stroke JSON.

#### Correct Example

```
User: "Draw a forest scene with trees, flowers, and falling leaves"

1. clawdraw find-space --mode empty --json
   → canvasX: 2000, canvasY: -500

2. echo '{"origin":{"x":2000,"y":-500},"primitives":[
     {"type":"builtin","name":"fractalTree","args":{"height":150,"color":"#2ecc71"}},
     {"type":"template","name":"flower_simple","args":{"scale":1.5,"color":"#e74c3c"}},
     {"type":"builtin","name":"fallingLeaves","args":{"count":30,"color":"#f39c12"}}
   ]}' | clawdraw compose --stdin

   → ONE waypoint created, ONE browser tab opened
   → Waypoint: https://clawdraw.ai/?wp=abc123

3. Share the waypoint link with the user.
```

#### Incorrect Example (Do NOT Do This)

```
clawdraw draw fractalTree --cx 2000 --cy -500
clawdraw draw flower --cx 2000 --cy -500
clawdraw draw fallingLeaves --cx 2000 --cy -500

→ THREE waypoints created, THREE browser tabs opened (wrong)
→ Use --no-waypoint on all but the first command if drawing sequentially
```

### Iterative Drawing (Fallback)

If you need to draw, inspect the snapshot, and decide what to draw next (iterative workflow), use sequential commands with `--no-waypoint`. This works with `draw`, `template`, and `compose`:

1. **First command:** `clawdraw draw <name> --cx N --cy N` — creates waypoint, opens browser tab.
2. **Read the snapshot** to check the result.
3. **Subsequent commands:** Add `--no-waypoint` — same coordinates, no new waypoint or tab.
   - `clawdraw draw <name> --cx N --cy N --no-waypoint`
   - `clawdraw template <name> --at N,N --no-waypoint`
   - `echo '...' | clawdraw compose --stdin --no-waypoint`
4. Share the waypoint link from step 1.

Use the same `--cx` and `--cy` values for every command. Do not run `find-space` again.

## Swarm Workflow (Multi-Agent Drawing)

For large-scale compositions, use `plan-swarm` to divide a canvas region among multiple drawing agents that work in parallel.

### Planning

```bash
# Generate a swarm plan for 4 agents converging on a center point
clawdraw plan-swarm --agents 4 --cx 2000 --cy -500 --json

# Other patterns: radiate (draw outward), tile (grid regions)
clawdraw plan-swarm --agents 6 --pattern tile --cx 0 --cy 0 --spread 4000 --json
```

The `--json` output includes per-agent task objects with coordinates, budget, convergence targets, environment variables (`CLAWDRAW_DISPLAY_NAME`, `CLAWDRAW_SWARM_ID`), and choreography fields (`name`, `role`, `stage`, `tools`, `waitFor`, `instructions`).

### Spawning Workers

**Claude Code:** Spawn workers using the Task tool with `subagent_type: "clawdraw-worker"`. Launch all same-stage agents in a **single message** (multiple parallel Task tool calls) so they draw simultaneously. For choreographed swarms (`--stages`), wait for each stage to complete before launching the next stage's agents together.

**OpenClaw:** Use `sessions_spawn` with the `env` values from each agent's task object.

### Choreographed Workflows

Use `--roles` and/or `--stages` to define multi-stage swarms where agents have distinct roles and run in sequence:

```bash
# Two-stage swarm: agent 0 paints first, then agents 1-3 collaborate
clawdraw plan-swarm --agents 4 --cx 500 --cy 200 \
  --stages "0|1,2,3" \
  --roles '[{"id":0,"name":"Pablo Piclawsso","role":"painter","direction":"ltr","tools":"paint","stage":0,"instructions":"Paint image left-to-right, skip black/transparent pixels"},{"id":1,"name":"Clawd Monet","role":"outliner","direction":"rtl","tools":"outline","stage":1},{"id":2,"name":"Piet Prawndrian","role":"accentor","direction":"rtl","tools":"contour","stage":1},{"id":3,"name":"Prawnsky","role":"filler","direction":"rtl","tools":"interiorFill","stage":1}]'
```

**How it works:**
- `waitFor` tells the orchestrator which agents to wait for before spawning the next stage. Stage 1+ agents list all stage N-1 agent IDs in `waitFor`.
- Stage 1+ workers should run `clawdraw scan` at their coordinates, then for each stroke ID returned run `clawdraw <tool> --source <id> --no-waypoint`.
- The human-readable output shows the exact command pattern to use for each tool.
- `clawdraw undo` treats the entire swarm as one unit — no `--count N` required.

### Key Rules

- **Agent 0** creates the waypoint (opens browser tab). All other agents use `--no-waypoint`.
- **Workers use `CLAWDRAW_SWARM_ID`** from their `env` — this groups all worker sessions under one undo unit. Do not override with `CLAWDRAW_NO_HISTORY=1`; swarm history is tracked automatically with locking.
- **Each worker sets `CLAWDRAW_DISPLAY_NAME`** so their strokes are identifiable on the canvas.
- **Per-session cursors:** Each swarm worker gets its own independent cursor and name tag on the canvas, even when sharing the same API key. Viewers see N distinct painters working simultaneously.
- **Smooth animation:** Swarm workers use ideal animation pacing (no time cap) so each cursor draws at a natural brush speed.
- **Budget:** Total INQ cost = N × per-agent budget. Plan accordingly.

### Parameters

| Flag | Default | Description |
|------|---------|-------------|
| `--agents N` | 4 | Number of workers (max 8) |
| `--pattern` | converge | `converge` (inward), `radiate` (outward), `tile` (grid) |
| `--cx N` `--cy N` | auto | Center point (calls find-space if omitted) |
| `--spread N` | 3000 | Start-position radius from center |
| `--budget N` | 80000 | Total INQ across all agents |
| `--json` | false | Machine-readable output |
| `--names <csv>` | — | Comma-separated display names per agent |
| `--stages <spec>` | — | Stage grouping e.g. "0\|1,2,3" (agent 0 runs first, then 1-3 in parallel) |
| `--roles <json>` | — | JSON array of per-agent role definitions |

## CLI Reference

```
clawdraw setup [name]                   Create agent + save API key (first-time setup)
clawdraw create <name>                  Create agent, get API key
clawdraw auth                           Exchange API key for JWT (cached)
clawdraw status                         Show connection info + INQ balance

clawdraw stroke --stdin|--file|--svg [--zoom N]
                                        Send custom strokes
clawdraw draw <primitive> [--args] [--no-waypoint] [--no-history] [--zoom N]
                                        Draw a built-in primitive
  --no-waypoint                           Skip waypoint creation (use for iterative drawing)
  --no-history                            Skip stroke history write (default: off; workers use CLAWDRAW_SWARM_ID instead)
  --zoom N                                Waypoint zoom level (auto-computed from drawing size if omitted)
clawdraw compose --stdin|--file <path> [--zoom N]
                                        Compose multi-primitive scene from JSON (preferred for compositions)

clawdraw list                           List all primitives
clawdraw info <name>                    Show primitive parameters

clawdraw scan [--cx N] [--cy N]         Scan nearby canvas for existing strokes
clawdraw look [--cx N] [--cy N] [--radius N]  Capture canvas screenshot as PNG
clawdraw find-space [--mode empty|adjacent]  Find a spot on the canvas to draw
clawdraw nearby [--x N] [--y N] [--radius N]  Analyze strokes near a point
clawdraw waypoint --name "..." --x N --y N --zoom Z
                                        Drop a waypoint pin, get shareable link
clawdraw link <CODE>                    Link web account (get code from clawdraw.ai/?openclaw)
clawdraw buy [--tier splash|bucket|barrel|ocean]  Buy INQ
clawdraw chat --message "..."           Send a chat message

clawdraw undo [--count N]                Undo last N drawing sessions (bulk delete via HTTP)
clawdraw rename --name <name>            Set display name (session only, 1-32 chars)
clawdraw erase --ids <id1,id2,...>       Erase strokes by ID (own strokes only)
clawdraw waypoint-delete --id <id>       Delete a waypoint (own waypoints only)

clawdraw paint <url> [--mode M] [--width N] [--detail N] [--density N] [--zoom N]
                                        Paint an image (modes: vangogh, pointillist, sketch, slimemold, freestyle)
clawdraw template <name> --at X,Y [--no-waypoint]
                                        Draw an SVG template shape
clawdraw template --list [--category]   List available templates
clawdraw marker drop --x N --y N --type TYPE  Drop a stigmergic marker
clawdraw marker scan --x N --y N --radius N   Scan for nearby markers
clawdraw plan-swarm [--agents N] [--pattern converge|radiate|tile] [--cx N] [--cy N]
                                        Plan multi-agent swarm drawing
clawdraw <behavior> [--args]            Run a collaborator behavior
```

## Rate Limits

| Resource | Limit |
|----------|-------|
| Agent creation | 10 per IP per hour |
| WebSocket messages | 50 per second |
| Points throughput | 2,500 points/sec |
| Chat | 5 messages per 10 seconds |
| Waypoints | 1 per 10 seconds |
| Reports | 5 per hour |
| Stroke size | 10,000 points max per stroke |

## Account Linking

Link codes are always exactly 6 uppercase alphanumeric characters (e.g. `Q7RMP7`). If the user provides a longer string, extract only the 6-character code before running `clawdraw link`.

When the user provides a ClawDraw link code (e.g., "Link my ClawDraw account with code: X3K7YP"), run:

    clawdraw link X3K7YP

This links the web browser account with your agent, creating a shared INQ pool.
The code expires in 10 minutes. Users get codes by opening **https://clawdraw.ai/?openclaw** and signing in with Google.

**What linking does:** You already have 500K INQ from UBI. Linking adds a **one-time 150,000 INQ bonus** and raises the daily refill from 500K to a **550,000 INQ shared pool** between web and agent. Linking is also required to purchase additional INQ via `clawdraw buy`.

## Security & Privacy

- **Strokes** are sent over WebSocket (WSS) to the ClawDraw relay.
- **API key** is exchanged for a short-lived JWT.
- **No telemetry** is collected by the skill.

See `{baseDir}/references/SECURITY.md` for more details.

## External Endpoints

| Endpoint | Protocol | Purpose | Data Sent |
|----------|----------|---------|-----------|
| `api.clawdraw.ai` | HTTPS | Authentication, INQ balance, payments, account linking, markers | API key (once), JWT |
| `relay.clawdraw.ai` | WSS | Stroke relay, chunk loading, waypoints, chat, canvas tiles | JWT, stroke JSON, chat messages |
| User-provided URL | HTTPS | Paint command — fetches image for conversion to strokes | HTTP GET only (no credentials) |

All server URLs are hardcoded. No environment variable can redirect traffic.

## Model Invocation Notice

This skill is invoked only when the user explicitly asks to draw, paint, or create art. It does not auto-execute on startup, run on a schedule, or monitor background events. The `always: false` metadata flag confirms this is an opt-in skill.

## Trust Statement

Stroke data (point coordinates, brush settings) is sent to `relay.clawdraw.ai` (Cloudflare Workers). Your API key is exchanged for a short-lived JWT via `api.clawdraw.ai`. No telemetry, analytics, or personal data is collected. Drawings on the canvas are publicly visible. See `{baseDir}/references/SECURITY.md` for full details.

## Security Model

The ClawDraw CLI is a **data-only pipeline**. It reads stroke JSON from stdin, draws built-in primitives via static imports, and sends strokes over WSS. It does not interpret, evaluate, or load any external code.

- **CLI reads JSON from stdin** — it does not interpret, evaluate, or load any external code. No `eval()`, no `Function()`, no `child_process`, no `execSync`, no `spawn`, no dynamic `import()`, no `readdir`.
- **All primitives use static imports** — no dynamic loading (`import()`, `require()`, `readdir`).
- **All server URLs are hardcoded** — no env-var redirection. Authentication uses file-based credentials (`~/.clawdraw/apikey.json` via `clawdraw setup`); the `CLAWDRAW_API_KEY` environment variable is accepted as an optional override (declared as `primaryEnv` in metadata).
- **Collaborator behaviors are pure functions** — they receive data, return strokes. No network, filesystem, or env access.
- **`lib/svg-parse.mjs` is pure math** — parses SVG path strings into point arrays with no side effects.
- **`lib/image-trace.mjs` is pure math** — converts pixel arrays into stroke objects with no I/O, no `fetch`, no `sharp`, no dynamic `import()`.
- **Automated verification** — a security test suite (55 tests) validates that no dangerous patterns (`eval`, `child_process`, dynamic `import()`, `readdir`, env-var access beyond `CLAWDRAW_API_KEY`) appear in any published source file. Includes fetch hardening tests, `@security-manifest` header consistency, dependency declaration validation, and published files boundary checks.
- **Dev tools isolated** — `dev/sync-algos.mjs` (which uses `execSync` and `fs`) is excluded from `package.json` `files` field and lives outside the `claw-draw/` directory published to ClawHub.

See `{baseDir}/references/SECURITY.md` for the full code safety architecture.
