/**
 * ClawAudit — System Auditor Phase 1 Tests
 * Tests for Docker Security, Process Isolation, and Network Segmentation checks
 */

import { test } from "node:test";
import { strict as assert } from "node:assert";
import { resolve, dirname } from "path";
import { fileURLToPath } from "url";
import { createScriptRunner } from "./lib/test-utils.mjs";

const __dirname = dirname(fileURLToPath(import.meta.url));
const AUDIT_SCRIPT = resolve(__dirname, "..", "scripts", "audit-system.mjs");
const runAudit = createScriptRunner(AUDIT_SCRIPT);

// Run audit once and share result across all tests (each test only inspects the output)
const data = runAudit();

// =============================================================================
// Docker Security Tests (SYS-070, SYS-071, SYS-072)
// =============================================================================

test("SYS-070: Docker daemon TCP exposure check exists", () => {
  // Test runs without error - actual finding depends on Docker installation
  assert.ok(data.findings !== undefined, "Audit should return findings array");
  assert.ok(typeof data.totalChecks === "number", "Should have totalChecks count");
});

test("SYS-071: Privileged container check exists", () => {

  // If Docker is installed and has privileged containers, should detect
  // Otherwise no finding - that's OK
  assert.ok(data.findings !== undefined);
});

test("SYS-072: Docker socket mount check exists", () => {

  // If Docker is installed and has socket mounts, should detect
  assert.ok(data.findings !== undefined);
});

// =============================================================================
// Process Isolation Tests (SYS-080, SYS-081, SYS-082)
// =============================================================================

test("SYS-080: Root process detection check exists", () => {

  // If OpenClaw/clawdbot/moltbot is running as root, should detect
  assert.ok(data.findings !== undefined);
});

test("SYS-081: Host PID namespace sharing check exists", () => {

  // If containers share host PID namespace, should detect
  assert.ok(data.findings !== undefined);
});

test("SYS-082: Resource limits check exists", () => {

  // If OpenClaw processes have unlimited resources, should detect
  assert.ok(data.findings !== undefined);
});

// =============================================================================
// Network Segmentation Tests (SYS-100, SYS-101, SYS-102)
// =============================================================================

test("SYS-100: Cloud metadata service check exists", () => {

  // If running in cloud and metadata service is reachable, should detect
  assert.ok(data.findings !== undefined);
});

test("SYS-101: Egress filtering check exists", () => {

  // If no egress filtering configured, should report info
  assert.ok(data.findings !== undefined);
});

test("SYS-102: DNS configuration check exists", () => {

  // If using public DNS, should report info
  assert.ok(data.findings !== undefined);
});

// =============================================================================
// Integration Tests
// =============================================================================

test("audit-system.mjs: Phase 1 checks integrate without errors", () => {


  // Verify structure
  assert.ok(typeof data.critical === "number", "Should have critical count");
  assert.ok(typeof data.warnings === "number", "Should have warnings count");
  assert.ok(typeof data.info === "number", "Should have info count");
  assert.ok(Array.isArray(data.findings), "findings must be an array");

  // Verify all findings have required fields
  for (const f of data.findings) {
    assert.ok(typeof f.severity === "string", "finding must have severity");
    assert.ok(typeof f.code === "string", "finding must have code");
    assert.ok(typeof f.title === "string", "finding must have title");
    assert.ok(typeof f.detail === "string", "finding must have detail");
  }
});

test("audit-system.mjs: totalChecks increased after Phase 1", () => {


  // Phase 1 adds 3 new check functions (Docker, Process Isolation, Network Segmentation)
  // Previous checks were ~13, now should be at least 16
  assert.ok(data.totalChecks >= 16, `Expected at least 16 checks, got ${data.totalChecks}`);
});

test("audit-system.mjs: new check codes follow SYS-0XX pattern", () => {


  const phase1Codes = ["SYS-070", "SYS-071", "SYS-072",
                       "SYS-080", "SYS-081", "SYS-082",
                       "SYS-100", "SYS-101", "SYS-102"];

  const foundCodes = data.findings.map(f => f.code).filter(c => phase1Codes.includes(c));

  // At minimum, checks should execute even if no findings
  // We can't assert specific findings because they depend on system state
  assert.ok(true, "Phase 1 check codes are valid");
});
