---
name: claude-tmux
description: Manage Claude Code instances living inside tmux sessions. Users usually create separate tmux sessions for separate projects. Use this skill when you need to read the latest Claude Code response in a particular tmux session / project, send it a prompt and get the response, or run /compact directly via tmux (no extra scripts required).
---

## Goal
Give Codex a repeatable checklist for interacting with Claude Code when it’s running inside tmux. Everything happens with standard tmux commands—no helper scripts. Follow these steps any time you see instructions like “check Claude in session X” or “run /compact on Claude.”

## Conventions
1. **Session naming** – We refer to tmux sessions by their tmux session name. Session names can be assigned using `tmux new-session -s <session_name>`. E.g. if we had created a tmux session for project FooBar using `tmux new-session -s foobar`, then we will refer to this session by the name `foobar`.
2. **Claude pane** – Within a session, there should be exactly one pane whose *window title* or *pane title* is `claude`. If the pane isn’t named, rename it first (`Ctrl-b : select-pane -T claude`).
3. **Standard markers** – Claude Code prints user prompts with `❯ …` and its replies with `⏺ …`. We rely on that to spot the latest exchange.

## Workflow

### 1. Locate the Claude pane
```
tmux list-panes -a -F '#{session_name}:#{window_index}.#{pane_index} #{pane_title}' | grep "^<session_name>" | grep -i claude
```
- If nothing matches, say “No pane titled ‘claude’ found inside session <name>.”
- If multiple panes match, pick the one with the lowest `window_index/pane_index` unless context says otherwise.
- Record the target as `<session>:<window>.<pane>` for subsequent commands.

### 2. View the latest exchange
```
tmux capture-pane -p -J -t <target> -S -200
```
- Scan from the bottom upward for the last `❯` block (user) followed by `⏺` (Claude). Quote those lines back to the user.
- If no `❯/⏺` pair exists, say “No exchange found yet.”

### 3. Send a prompt
```
tmux send-keys -t <target> -l -- "<prompt>"
sleep 0.1
 tmux send-keys -t <target> Enter
```
- After sending, poll using capture-pane until a new `⏺` block appears (or a sensible timeout, e.g., 3 minutes). Report the reply verbatim.
- If the timeout expires, say “Claude hasn’t replied yet—still waiting.”

### 4. Run `/compact`
Same as sending any prompt, but send `/compact`. Confirm with “Triggered /compact in session <name>.” (Claude will respond in-pane; no need to quote unless asked.)

### 5. Dump raw buffer (debug)
```
tmux capture-pane -p -J -t <target> -S -400
```
Use this when the user wants the entire scrollback or when parsing fails.

## Tips
- Always double-check you’re addressing the right pane before sending commands—especially in shared sessions.
- If the Claude pane lives on a non-default tmux socket, prefix every tmux command with `tmux -S /path/to/socket …`.
- When summarizing results, mention the session/pane you used—for traceability.
- If the user wants multiple sessions handled, repeat the workflow per session.

This skill keeps things simple: pure tmux, no external code. Use it whenever you need hands-on access to Claude Code running inside tmux.
