#!/bin/bash
# Watchdog — watches the watchman.
#
# Runs independently of Claude Code (cron, systemd timer, or agent harness).
# Checks all supervised sessions in supervisor-state.json:
#   - Is the tmux session still alive?
#   - Is Claude Code still running inside it?
#   - Has anything changed since last check?
#
# If a session looks dead with no hook having fired, notifies.
# Does NOT use claude -p or any LLM — pure bash, zero dependencies beyond tmux + jq.
#
# Usage:
#   watchdog.sh [state-file]
#
# Designed to be called by:
#   - cron (system): */15 * * * * /path/to/watchdog.sh
#   - OpenClaw cron: systemEvent "cc-supervisor: watchdog"
#   - systemd timer
#   - anything that runs periodically

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
source "$SCRIPT_DIR/lib.sh"

STATE_FILE="${1:-${CCS_STATE_FILE:-${HOME}/.openclaw/workspace/supervisor-state.json}}"
LOG_FILE="${CCS_LOG_FILE:-/tmp/ccs-triage.log}"

if [ ! -f "$STATE_FILE" ]; then
  exit 0  # No supervised sessions, nothing to do
fi

# Get all running sessions
SESSIONS=$(jq -r '.sessions | to_entries[] | select(.value.status == "running") | .key' "$STATE_FILE" 2>/dev/null)

if [ -z "$SESSIONS" ]; then
  exit 0  # No active sessions
fi

NOW=$(date +%s)

for SESSION_NAME in $SESSIONS; do
  SOCKET=$(jq -r --arg s "$SESSION_NAME" '.sessions[$s].socket // ""' "$STATE_FILE")
  TMUX_SESSION=$(jq -r --arg s "$SESSION_NAME" '.sessions[$s].tmuxSession // ""' "$STATE_FILE")
  PROJECT_DIR=$(jq -r --arg s "$SESSION_NAME" '.sessions[$s].projectDir // ""' "$STATE_FILE")
  GOAL=$(jq -r --arg s "$SESSION_NAME" '.sessions[$s].goal // "unknown"' "$STATE_FILE")
  STARTED_AT=$(jq -r --arg s "$SESSION_NAME" '.sessions[$s].startedAt // ""' "$STATE_FILE")
  ESCALATE_AFTER=$(jq -r --arg s "$SESSION_NAME" '.sessions[$s].escalateAfterMin // 60' "$STATE_FILE")
  LAST_CHECK=$(jq -r --arg s "$SESSION_NAME" '.sessions[$s].lastWatchdogAt // ""' "$STATE_FILE")

  CONFIG=$(ccs_find_config "$PROJECT_DIR")
  DEAD_REASON=""

  # Check 1: Does the tmux socket exist?
  if [ ! -S "$SOCKET" ]; then
    DEAD_REASON="tmux socket gone ($SOCKET)"
  # Check 2: Does the tmux session exist?
  elif ! tmux -S "$SOCKET" has-session -t "$TMUX_SESSION" 2>/dev/null; then
    DEAD_REASON="tmux session '$TMUX_SESSION' no longer exists"
  else
    # Check 3: Is Claude Code still running? Look for the process.
    PANE_PID=$(tmux -S "$SOCKET" list-panes -t "$TMUX_SESSION" -F '#{pane_pid}' 2>/dev/null | head -1)
    if [ -n "$PANE_PID" ]; then
      # Check if any claude process is a descendant of the pane
      if ! pgrep -P "$PANE_PID" -f "claude" >/dev/null 2>&1; then
        # No claude process — check if shell prompt is back
        PANE_OUTPUT=$(tmux -S "$SOCKET" capture-pane -p -J -t "$TMUX_SESSION" -S -10 2>/dev/null || echo "")
        if echo "$PANE_OUTPUT" | tail -3 | grep -qE '^\$ |^❯ |^% '; then
          DEAD_REASON="Claude Code exited (shell prompt returned, no hooks fired)"
        fi
        # If no prompt either, might be some other process — don't flag
      else
        # Claude is running — check if it's idle at the prompt (waiting for input)
        PANE_OUTPUT=$(tmux -S "$SOCKET" capture-pane -p -J -t "$TMUX_SESSION" -S -15 2>/dev/null || echo "")
        PANE_STATUS=$(ccs_parse_status "$PANE_OUTPUT")
        if [ "$PANE_STATUS" = "IDLE" ]; then
          IDLE_TIMEOUT=$(ccs_config_get "$CONFIG" "idle.timeout" "120")
          IDLE_MSG=$(ccs_config_get "$CONFIG" "idle.nudge_message" "continue with the task")
          IDLE_MARKER="/tmp/ccs-idle-${SESSION_NAME}"

          if [ -f "$IDLE_MARKER" ]; then
            IDLE_SINCE=$(cat "$IDLE_MARKER")
            IDLE_SECS=$(( NOW - IDLE_SINCE ))
            if [ "$IDLE_SECS" -ge "$IDLE_TIMEOUT" ]; then
              # Idle too long — nudge via send-keys (two-step pattern)
              tmux -S "$SOCKET" send-keys -t "$TMUX_SESSION" "$IDLE_MSG" 2>/dev/null || true
              tmux -S "$SOCKET" send-keys -t "$TMUX_SESSION" Enter 2>/dev/null || true
              echo "[$(date -u +%FT%TZ)] IDLE_NUDGE | $SESSION_NAME | idle ${IDLE_SECS}s" >> "$LOG_FILE"
              rm -f "$IDLE_MARKER"
            fi
          else
            # First time seeing idle — record timestamp
            echo "$NOW" > "$IDLE_MARKER"
          fi
        else
          # Not idle — clear marker
          rm -f "/tmp/ccs-idle-${SESSION_NAME}"
        fi
      fi
    fi
  fi

  # Check 4: Duration exceeded?
  if [ -z "$DEAD_REASON" ] && [ -n "$STARTED_AT" ]; then
    START_EPOCH=$(date -d "$STARTED_AT" +%s 2>/dev/null || echo 0)
    if [ "$START_EPOCH" -gt 0 ]; then
      ELAPSED_MIN=$(( (NOW - START_EPOCH) / 60 ))
      if [ "$ELAPSED_MIN" -gt "$ESCALATE_AFTER" ]; then
        # Don't mark dead, but note duration for context
        DURATION_NOTE="running ${ELAPSED_MIN}min (limit: ${ESCALATE_AFTER}min)"
      fi
    fi
  fi

  # Update lastWatchdogAt
  TMP=$(mktemp)
  jq --arg s "$SESSION_NAME" --arg ts "$(date -u +%FT%TZ)" \
    '.sessions[$s].lastWatchdogAt = $ts' "$STATE_FILE" > "$TMP" && mv "$TMP" "$STATE_FILE"

  if [ -n "$DEAD_REASON" ]; then
    # Session is dead and hooks didn't catch it — notify
    echo "[$(date -u +%FT%TZ)] WATCHDOG | $SESSION_NAME | $DEAD_REASON" >> "$LOG_FILE"

    # Capture last output if possible
    LAST_OUTPUT=""
    if [ -S "$SOCKET" ] && tmux -S "$SOCKET" has-session -t "$TMUX_SESSION" 2>/dev/null; then
      LAST_OUTPUT=$(tmux -S "$SOCKET" capture-pane -p -J -t "$TMUX_SESSION" -S -15 2>/dev/null || echo "(could not capture)")
    fi

    ccs_notify "$CONFIG" \
      "cc-supervisor: WATCHDOG | session=$SESSION_NAME | reason=$DEAD_REASON | goal=$GOAL | last_output=$LAST_OUTPUT"

    # Mark session as needing attention
    TMP=$(mktemp)
    jq --arg s "$SESSION_NAME" '.sessions[$s].status = "watchdog-alert"' "$STATE_FILE" > "$TMP" && mv "$TMP" "$STATE_FILE"
  fi

done
