#!/bin/bash
# Install Claude Code Supervisor hooks into a project.
# Usage: install-hooks.sh [project-dir]
#
# - Copies hook scripts to .claude/hooks/supervisor/
# - Merges hook config into .claude/settings.json
# - Creates config file if it doesn't exist

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
REPO_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
PROJECT_DIR="${1:-.}"
PROJECT_DIR="$(cd "$PROJECT_DIR" && pwd)"

CLAUDE_DIR="$PROJECT_DIR/.claude"
HOOKS_DST="$CLAUDE_DIR/hooks/supervisor"
SETTINGS="$CLAUDE_DIR/settings.json"

echo "Installing Claude Code Supervisor into: $PROJECT_DIR"
echo ""

# Copy supervisor scripts (hooks + lib + triage)
mkdir -p "$HOOKS_DST"
cp "$REPO_DIR/scripts/hooks/on-stop.sh" "$HOOKS_DST/"
cp "$REPO_DIR/scripts/hooks/on-error.sh" "$HOOKS_DST/"
cp "$REPO_DIR/scripts/hooks/on-notify.sh" "$HOOKS_DST/"
cp "$REPO_DIR/scripts/triage.sh" "$HOOKS_DST/"
cp "$REPO_DIR/scripts/lib.sh" "$HOOKS_DST/"
chmod +x "$HOOKS_DST"/*.sh
echo "  ✓ Copied supervisor scripts to $HOOKS_DST/"

# Create project config if missing
if [ ! -f "$PROJECT_DIR/.claude-code-supervisor.yml" ]; then
  cp "$REPO_DIR/supervisor.yml.example" "$PROJECT_DIR/.claude-code-supervisor.yml"
  echo "  ✓ Created .claude-code-supervisor.yml (edit to configure)"
else
  echo "  · .claude-code-supervisor.yml already exists, skipping"
fi

# Build hooks config — paths relative to project root
HOOKS_CONFIG=$(cat <<'HOOKSJSON'
{
  "Stop": [
    {
      "matcher": "",
      "hooks": [
        {
          "type": "command",
          "command": ".claude/hooks/supervisor/on-stop.sh"
        }
      ]
    }
  ],
  "PostToolUseFailure": [
    {
      "matcher": "",
      "hooks": [
        {
          "type": "command",
          "command": ".claude/hooks/supervisor/on-error.sh"
        }
      ]
    }
  ],
  "Notification": [
    {
      "matcher": "",
      "hooks": [
        {
          "type": "command",
          "command": ".claude/hooks/supervisor/on-notify.sh"
        }
      ]
    }
  ]
}
HOOKSJSON
)

# Merge into settings.json
mkdir -p "$CLAUDE_DIR"
if [ -f "$SETTINGS" ]; then
  EXISTING=$(cat "$SETTINGS")
  echo "$EXISTING" | jq --argjson hooks "$HOOKS_CONFIG" '.hooks = (.hooks // {}) + $hooks' > "$SETTINGS.tmp"
  mv "$SETTINGS.tmp" "$SETTINGS"
  echo "  ✓ Merged hooks into $SETTINGS"
else
  echo "{\"hooks\": $HOOKS_CONFIG}" | jq '.' > "$SETTINGS"
  echo "  ✓ Created $SETTINGS"
fi

# Generate notify wrapper script for agent use
source "$REPO_DIR/scripts/lib.sh"
CONFIG=$(ccs_find_config "$PROJECT_DIR")
ccs_generate_notify_script "$CONFIG" "/tmp/supervisor-notify.sh"
echo "  ✓ Generated /tmp/supervisor-notify.sh (agent can call this on completion)"

echo ""
echo "Done! Next steps:"
echo ""
echo "  1. Edit .claude-code-supervisor.yml to configure triage model + notify command"
echo "  2. Register supervised sessions in ~/. openclaw/workspace/supervisor-state.json"
echo "     (see: $REPO_DIR/assets/supervisor-state.template.json)"
echo "  3. Launch Claude Code in tmux — hooks will triage and notify automatically"
echo ""
echo "Test a hook manually:"
echo "  echo '{\"stop_reason\":\"end_turn\",\"session_id\":\"test\",\"cwd\":\"$PROJECT_DIR\"}' | $HOOKS_DST/on-stop.sh"
