#!/bin/bash
# Claude Code Notification hook — fires when Claude is waiting for input.
#
# Option D: bash pre-filter by notification type.
#
# - idle_prompt → triage (agent waiting, might need nudge)
# - permission_prompt → always triage (might need approval)
# - auth_* → skip (internal, transient)
# - other → triage

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
SUPERVISOR_DIR="${SCRIPT_DIR%/hooks}"

INPUT=$(cat)

NOTIFY_TYPE=$(echo "$INPUT" | jq -r '.type // "unknown"')
SESSION_ID=$(echo "$INPUT" | jq -r '.session_id // "unknown"')
CWD=$(echo "$INPUT" | jq -r '.cwd // "unknown"')
MESSAGE=$(echo "$INPUT" | jq -r '.message // ""' | head -c 300)

case "$NOTIFY_TYPE" in
  auth_*)
    # Auth events are transient, skip
    exit 0
    ;;
  permission_prompt)
    # Agent needs permission — might be able to auto-approve, or escalate
    "$SUPERVISOR_DIR/triage.sh" "notification:permission" "$CWD" \
      "Permission requested: $MESSAGE | Session: $SESSION_ID" &
    ;;
  idle_prompt)
    # Agent is idle, waiting for human input
    "$SUPERVISOR_DIR/triage.sh" "notification:idle" "$CWD" \
      "Agent idle, waiting for input. Message: $MESSAGE | Session: $SESSION_ID" &
    ;;
  *)
    "$SUPERVISOR_DIR/triage.sh" "notification:$NOTIFY_TYPE" "$CWD" \
      "Type: $NOTIFY_TYPE | Message: $MESSAGE | Session: $SESSION_ID" &
    ;;
esac

exit 0
