# State Detection Patterns

When assessing a Claude Code tmux session, capture the last 100-200 lines and
look for these patterns in order of priority.

## Terminal Patterns

### Agent Finished (prompt returned)
```
❯ _
```
Shell prompt is back. Agent exited. Check the output above for success/failure.

### API Error (transient)
```
API Error: 500 {"type":"error","error":{"type":"api_error","message":"Internal server error"}}
```
**Action:** Nudge with "continue, the API error was transient — pick up where you left off."

### Rate Limited
```
API Error: 429
rate_limit_exceeded
```
**Action:** Wait 60s, then nudge. If repeated, escalate.

### Context Limit / Max Tokens
```
max_tokens
context window
```
**Action:** The session may need a `/compact` or fresh start. Escalate if complex.

### Permission Prompt
```
Allow this action? (y/n)
Do you want to proceed?
bypass permissions
```
**Action:** If within safety bounds (e.g. file writes in project dir), nudge with "y".
Otherwise escalate to human.

### Stuck in Loop
Same error or same action repeated 3+ times in the last 50 lines.
**Action:** Escalate. Agent is not making progress.

### Test Results
```
✓ passed
✗ failed
FAIL
All \d+ tests passed
```
Check for pass/fail. If all pass and goal was testing, mark done.

### Git Operations
```
[branch commit_hash] commit message
→ origin/branch
```
Committed and/or pushed. If goal was "commit and push," mark done.

## Tmux send-keys Pattern

When sending input to Claude Code via tmux, text entry and submission are
separate steps. Embedding `\n` or `\r` in the text does NOT submit in Claude
Code's TUI.

```bash
# Step 1: Enter the text
tmux -S "$SOCKET" send-keys -t "$SESSION" "continue with the task"

# Step 2: Submit it
tmux -S "$SOCKET" send-keys -t "$SESSION" Enter
```

This two-step pattern applies to all tmux interactions with Claude Code: nudge
messages, prompt responses, and any text input. Always send `Enter` as a
separate `send-keys` call.

## Assessment Priority

1. Is the tmux session still alive? (If not → escalate)
2. Is there an API error in the last 10 lines? (→ nudge)
3. Is the shell prompt back? (→ assess completion)
4. Is Claude Code still running? (→ wait, it's working)
5. Is it waiting for input? (→ nudge or escalate based on context)
