---
name: claude-code-supervisor
description: >
  Supervise Claude Code sessions running in tmux. Uses Claude Code hooks with
  bash pre-filtering (Option D) and fast LLM triage to detect errors, stuck
  agents, and task completion. Harness-agnostic — works with OpenClaw, webhooks,
  ntfy, or any notification backend. Use when: (1) launching long-running Claude
  Code tasks that need monitoring, (2) setting up automatic nudging for API
  errors or premature stops, (3) getting progress reports from background coding
  agents, (4) continuing work after session/context limits reset.
  Requires: tmux, claude CLI.
metadata:
  {
    "openclaw":
      {
        "emoji": "👷",
        "os": ["darwin", "linux"],
        "requires": { "bins": ["tmux"], "anyBins": ["claude"] },
      },
  }
---

# Claude Code Supervisor

Bridge between Claude Code's lifecycle hooks and your agent harness.

## Architecture

```
Claude Code (in tmux)
  │  Stop / Error / Notification
  ▼
Bash pre-filter (Option D)
  │  obvious cases handled directly
  │  ambiguous cases pass through
  ▼
Fast LLM triage (claude -p with Haiku, or local LLM)
  │  classifies: FINE | NEEDS_NUDGE | STUCK | DONE | ESCALATE
  │  FINE → logged silently
  ▼
Notify command (configurable)
  │  openclaw wake, webhook, ntfy, script, etc.
  ▼
Agent harness decides + acts
  │  nudge (send-keys to tmux), wait, escalate to human
```

## Quick Start

### 1. Install hooks into a project

```bash
{baseDir}/scripts/install-hooks.sh /path/to/your/project
```

Creates:
- `.claude/hooks/supervisor/` — hook scripts + triage
- `.claude/settings.json` — wired into Claude Code lifecycle
- `.claude-code-supervisor.yml` — configuration (edit this)

### 2. Configure

Edit `.claude-code-supervisor.yml`:

```yaml
triage:
  command: "claude -p --no-session-persistence"  # or: ollama run llama3.2
  model: "claude-haiku-4-20250414"

notify:
  command: "openclaw gateway call wake --params"  # or: curl, ntfy, script
```

### 3. Register a supervised session

Create `~/.openclaw/workspace/supervisor-state.json` (or wherever your harness keeps state):

```json
{
  "sessions": {
    "my-task": {
      "socket": "/tmp/openclaw-tmux-sockets/openclaw.sock",
      "tmuxSession": "my-task",
      "projectDir": "/path/to/project",
      "goal": "Fix issue #42",
      "successCriteria": "Tests pass, committed",
      "maxNudges": 5,
      "escalateAfterMin": 60,
      "status": "running"
    }
  }
}
```

### 4. Launch Claude Code in tmux

```bash
SOCKET="/tmp/openclaw-tmux-sockets/openclaw.sock"
tmux -S "$SOCKET" new -d -s my-task
tmux -S "$SOCKET" send-keys -t my-task "cd /path/to/project && claude 'Fix issue #42'" Enter
```

Hooks fire automatically. Triage assesses. You get notified only when it matters.

## How the Pre-Filter Works (Option D)

Not every hook event needs an LLM call. Bash catches the obvious cases first:

### on-stop.sh
| Signal | Bash decision | LLM triage? |
|--------|--------------|-------------|
| `max_tokens` | Always needs attention | ✅ Yes |
| `end_turn` + shell prompt back | Agent might be done | ✅ Yes |
| `end_turn` + no prompt | Agent is mid-work | ❌ Skip |
| `stop_sequence` | Normal | ❌ Skip |

### on-error.sh
| Signal | Bash decision | LLM triage? |
|--------|--------------|-------------|
| API 429 / rate limit | Transient, will resolve | ❌ Log only |
| API 500 | Agent likely stuck | ✅ Yes |
| Other tool error | Unknown severity | ✅ Yes |

### on-notify.sh
| Signal | Bash decision | LLM triage? |
|--------|--------------|-------------|
| `auth_*` | Internal, transient | ❌ Skip |
| `permission_prompt` | Needs decision | ✅ Yes |
| `idle_prompt` | Agent waiting | ✅ Yes |

## Triage Classifications

The LLM returns one of:

| Verdict | Meaning | Typical action |
|---------|---------|----------------|
| **FINE** | Agent is working normally | Log silently, no notification |
| **NEEDS_NUDGE** | Transient error, should continue | Send "continue" to tmux |
| **STUCK** | Looping or not progressing | Try different approach or escalate |
| **DONE** | Task completed successfully | Report to human |
| **ESCALATE** | Needs human judgment | Notify human with context |

## Handling Notifications (for agent harness authors)

Wake events arrive with the prefix `cc-supervisor:` followed by the classification:

```
cc-supervisor: NEEDS_NUDGE | error:api_500 | cwd=/home/user/project | ...
cc-supervisor: DONE | stopped:end_turn:prompt_back | cwd=/home/user/project | ...
```

### Nudging via tmux

```bash
tmux -S "$SOCKET" send-keys -t "$SESSION" "continue — the API error was transient" Enter
```

### Escalation format

See `references/escalation-rules.md` for when to nudge vs escalate and quiet hours.

## Watchdog (Who Watches the Watchman?)

Hooks depend on Claude Code being alive. If the session hard-crashes, hits account
limits, or the process gets OOM-killed, no hooks fire. The watchdog catches this.

`scripts/watchdog.sh` is a pure bash script (no LLM, no Claude Code dependency) that:
1. Reads `supervisor-state.json` for all `"running"` sessions
2. Checks: is the tmux socket alive? Is the session there? Is Claude Code still running?
3. If something is dead and no hook reported it → notifies via the configured command
4. Updates `lastWatchdogAt` in state for tracking

Run it on a timer. Choose your poison:

**System cron:**
```bash
*/15 * * * * /path/to/claude-code-supervisor/scripts/watchdog.sh
```

**OpenClaw cron:**
```json
{
  "schedule": { "kind": "every", "everyMs": 900000 },
  "payload": { "kind": "systemEvent", "text": "cc-supervisor: watchdog — run /path/to/scripts/watchdog.sh and report" },
  "sessionTarget": "main"
}
```

**systemd timer, launchd, or whatever runs periodically on your box.**

The watchdog is deliberately dumb — no LLM, no complex logic, just "is the process still
there?" This means it works even when the triage model is down, the API is melting, or
your account hit its limit. Belts and suspenders.

## Files

- `scripts/install-hooks.sh` — one-command setup per project
- `scripts/hooks/on-stop.sh` — Stop event handler with bash pre-filter
- `scripts/hooks/on-error.sh` — PostToolUseFailure handler with bash pre-filter
- `scripts/hooks/on-notify.sh` — Notification handler with bash pre-filter
- `scripts/triage.sh` — LLM triage (called by hooks for ambiguous cases)
- `scripts/lib.sh` — shared config loading and notification functions
- `scripts/watchdog.sh` — dead session detector (pure bash, no LLM dependency)
- `references/state-patterns.md` — terminal output pattern matching guide
- `references/escalation-rules.md` — when to nudge vs escalate vs wait
- `supervisor.yml.example` — example configuration
