#!/usr/bin/env bash
set -euo pipefail

LABEL=""
WORKDIR=""
SESSION=""
SOCKET="${TMPDIR:-/tmp}/clawdbot-tmux-sockets/clawdbot.sock"
SEND_WAKE=true
LINT_CMD="npm run lint"
BUILD_CMD="npm run build"

while [[ $# -gt 0 ]]; do
  case "$1" in
    --label) LABEL="$2"; shift 2 ;;
    --workdir) WORKDIR="$2"; shift 2 ;;
    --session) SESSION="$2"; shift 2 ;;
    --socket) SOCKET="$2"; shift 2 ;;
    --no-wake) SEND_WAKE=false; shift ;;
    --lint-cmd) LINT_CMD="$2"; shift 2 ;;
    --build-cmd) BUILD_CMD="$2"; shift 2 ;;
    *) echo "Unknown arg: $1"; exit 1 ;;
  esac
done

if [[ -z "$LABEL" || -z "$WORKDIR" ]]; then
  echo "Usage: $0 --label <label> --workdir <dir> [--session cc-xxx] [--socket path] [--no-wake]"
  exit 1
fi

SESSION="${SESSION:-cc-${LABEL}}"
REPORT_JSON="/tmp/${SESSION}-completion-report.json"
REPORT_MD="/tmp/${SESSION}-completion-report.md"

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
WAKE_SCRIPT="$SCRIPT_DIR/wake.sh"

cd "$WORKDIR"

status_short="$(git status --short || true)"
changed_files="$(git diff --name-only || true)"
diff_stat="$(git diff --stat || true)"

lint_ok=true
lint_out="skipped"
if [[ -n "$LINT_CMD" ]]; then
  if ! lint_out="$($LINT_CMD 2>&1)"; then
    lint_ok=false
  fi
fi

build_ok=true
build_out="skipped"
if [[ -n "$BUILD_CMD" ]]; then
  if ! build_out="$($BUILD_CMD 2>&1)"; then
    build_ok=false
  fi
fi

risk="low"
recommendation="keep"
if [[ "$lint_ok" != true || "$build_ok" != true ]]; then
  risk="high"
  recommendation="partial_rollback"
fi

scope_drift=false

# Build JSON report using jq (no shell variable injection into code)
jq -n \
  --arg label "$LABEL" \
  --arg workdir "$WORKDIR" \
  --arg changed_files "$changed_files" \
  --arg diff_stat "$diff_stat" \
  --argjson lint_ok "$lint_ok" \
  --arg lint_out "${lint_out:0:4000}" \
  --argjson build_ok "$build_ok" \
  --arg build_out "${build_out:0:4000}" \
  --arg risk "$risk" \
  --argjson scope_drift "$scope_drift" \
  --arg recommendation "$recommendation" \
  '{
    label: $label,
    workdir: $workdir,
    changedFiles: ($changed_files | if . == "" then [] else split("\n") | map(select(. != "")) end),
    diffStat: $diff_stat,
    lint: { ok: $lint_ok, summary: $lint_out },
    build: { ok: $build_ok, summary: $build_out },
    risk: $risk,
    scopeDrift: $scope_drift,
    recommendation: $recommendation,
    notes: "Generated by complete-tmux-task.sh"
  }' > "$REPORT_JSON"

cat > "$REPORT_MD" <<EOF
# Claude Code Completion Report
- label: $LABEL
- workdir: $WORKDIR
- session: $SESSION
- report_json: $REPORT_JSON

## changed files

$changed_files

## diff stat

$diff_stat

## quality gates
- lint: $lint_ok
- build: $build_ok

## recommendation
- risk: $risk
- recommendation: $recommendation
- scope_drift: $scope_drift
EOF

echo "REPORT_JSON=$REPORT_JSON"
echo "REPORT_MD=$REPORT_MD"

if [[ "$SEND_WAKE" == true ]]; then
  bash "$WAKE_SCRIPT" "Claude Code done (${LABEL}) report=$REPORT_JSON" now
  echo "WAKE_SENT=true"
fi
