---
name: claude-code
description: Trigger Claude Code development tasks in observable tmux sessions with stable startup, progress visibility, and completion callback to OpenClaw. Use when user asks to run coding work via Claude Code and wants to SSH in, monitor progress, and get auto-notified for review after completion.
---

# Claude Code Orchestrator (tmux-first)

Use tmux-based orchestration for long coding tasks to avoid silent hangs and make progress observable.

## Standard workflow

1. Create prompt file (avoid long shell quote issues).
2. Start a dedicated tmux session.
3. Launch `claude --dangerously-skip-permissions` in interactive mode.
4. Paste prompt into Claude.
5. Require callback command in prompt (via wrapper):
   `bash {baseDir}/scripts/wake.sh "..." now`
6. Share socket/session attach command with user.
7. On completion, review diff + lint/build + risk summary.

## Start command

```bash
bash {baseDir}/scripts/start-tmux-task.sh \
  --label "gallery-detail-polish" \
  --workdir "/Users/yaxuan/.openclaw/workspace/work/active/02-gallery-ops" \
  --prompt-file "/Users/yaxuan/Downloads/gallery-website-design-system.md" \
  --task "参考这个修改我当前的画廊官网，注意优先打磨细节和质感，对整体结构展示先不用大改。"
```

## Monitor commands

```bash
# attach
bash {baseDir}/scripts/monitor-tmux-task.sh --attach --session <session>

# capture last 200 lines
bash {baseDir}/scripts/monitor-tmux-task.sh --session <session> --lines 200
```

## Task overview

List all running `cc-*` tasks at a glance — useful for "butler-style" summaries.

```bash
# Human-readable one-liner per task
bash {baseDir}/scripts/list-tasks.sh

# Structured JSON array (pipe to jq, feed to OpenClaw, etc.)
bash {baseDir}/scripts/list-tasks.sh --json | jq .
```

Options:
- `--lines <n>` — number of trailing pane lines to capture per task (default 20).
- `--socket <path>` — tmux socket path (default `$TMPDIR/clawdbot-tmux-sockets/clawdbot.sock`).
- `--json` — emit JSON array instead of human table.
- `--target ssh --ssh-host <alias>` — list sessions on a remote host.

Each entry contains: **label**, **session**, **status**, **sessionAlive**, **reportExists**, **reportJsonPath**, **lastLines**, **updatedAt**.

Combine with OpenClaw to generate a periodic butler summary:
```
# In an OpenClaw prompt / cron:
bash {baseDir}/scripts/list-tasks.sh --json | \
  openclaw gateway call summarize-tasks --stdin
```

## Rules

- Prefer interactive Claude in tmux for visibility (not long `claude -p` one-shot for large tasks).
- Always include callback via wrapper `bash {baseDir}/scripts/wake.sh "..." now` in prompt.
- Startup script now uses robust submit (ready-check + multi-Enter retry + execution-state detection) to avoid "prompt pasted but not submitted".
- If no pane output for >2-3 min, inspect and restart session.
- Kill stale Claude processes before restart.
- Always return: session name + attach command + current status.

## Status check (zero-token)

If wake not received within expected time, check task status before consuming tokens:

```bash
bash {baseDir}/scripts/status-tmux-task.sh --label <label>
```

Output: `STATUS=running|likely_done|stuck|idle|dead|done_session_ended`

- `likely_done` / `done_session_ended` → proceed to completion loop
- `running` → wait
- `stuck` → inspect (attach or capture-pane)
- `dead` → session lost, run complete-tmux-task.sh fallback
- `idle` → Claude may be waiting for input, inspect

## Completion loop (mandatory)

When wake event "Claude Code done (...)" arrives, complete this loop immediately:

1. Acknowledge user within 60s: "已收到完成信号，正在评估改动".
2. Preferred path: read completion report generated by Claude Code task:
   - `/tmp/cc-<label>-completion-report.json`
3. If report missing, run local fallback immediately:
   - `bash {baseDir}/scripts/complete-tmux-task.sh --label <label> --workdir <workdir>`
4. **Mandatory deep-read**: read full JSON/MD report before replying.
5. Read context before replying:
   - Read completion report file(s) (`/tmp/cc-<label>-completion-report.json/.md`)
   - Read recent tmux transcript (monitor script) to capture what Claude actually did/failed/tried
   - Incorporate the latest user constraints from current chat
6. Then provide assistant analysis (not a fixed template):
   - what was actually completed
   - what is reliable vs uncertain
   - key risks/tradeoffs in the user's context
   - concrete next-step options
7. Ask explicit decision from user if scope drift exists.

Do not stop at wake-only notification. Wake is trigger, not final delivery.

### Anti-pattern to avoid
- Forbidden: one-line fixed reply after wake without reading transcript + report.
- Forbidden: only relaying "done + report path" without analysis in user context.
- Forbidden: rigid templated output that ignores current conversation context.

## Hard guardrails added

- Prompt now enforces “no wake without report”:
  - task must write `/tmp/cc-<label>-completion-report.json` + `.md`
  - final wake must include `report=<json_path>`
- Recovery command exists for deterministic fallback:
  - `scripts/complete-tmux-task.sh` reproduces evidence and emits structured report
- Delivery SLA remains mandatory:
  - wake received -> ack <= 60s -> report
