---
name: cisco-asa-syslog
description: Parse, interpret, and analyze Cisco ASA (Adaptive Security Appliance) firewall syslog messages. Use this skill whenever working with Cisco ASA log files, syslog streams from ASA devices, firewall event analysis, or security investigations involving ASA-generated events. Covers the syslog protocol foundation and the ASA-specific message format with message ID categorization.
---

# Cisco ASA Syslog Analysis

This skill teaches you how to read, parse, and interpret log messages generated by Cisco ASA firewalls. All ASA logs are delivered via the syslog protocol, so understanding syslog is foundational before working with ASA-specific content.

---

## Part 1: Syslog Protocol Foundation

All Cisco ASA log messages are transported and formatted using the syslog standard (RFC 3164 / RFC 5424). You must understand syslog before interpreting ASA messages.

### Severity Levels

Every syslog message carries a numeric severity (0 = most critical, 7 = least critical):

| Level | Keyword       | Meaning                                  |
|-------|---------------|------------------------------------------|
| 0     | Emergency     | System is unusable                       |
| 1     | Alert         | Immediate action required                |
| 2     | Critical      | Critical conditions                      |
| 3     | Error         | Error conditions                         |
| 4     | Warning       | Warning conditions                       |
| 5     | Notice        | Normal but significant events            |
| 6     | Informational | Routine informational messages           |
| 7     | Debug         | Verbose diagnostic output                |

### Facility Codes

Facility codes indicate the source subsystem (0-23). Cisco ASA uses local4 (code 20) by default, but this is configurable on the device.

### PRI Value

The PRI header encodes both facility and severity:

  PRI = (Facility x 8) + Severity

Example: local4 (20) + Informational (6) -> (20 x 8) + 6 = 166, shown as <166> in the log header. This is the most common PRI value seen in ASA logs.

### RFC 3164 Format (Traditional)

  <PRI>TIMESTAMP HOSTNAME TAG: MESSAGE

Example:
  <166>Jan 15 10:22:01 fw01 : %ASA-6-302013: Built outbound TCP connection...

### RFC 5424 Format (ASA 9.10+)

Enabled via "logging timestamp rfc5424". Uses ISO 8601 UTC timestamps:

  <PRI>VERSION TIMESTAMP HOSTNAME APP-NAME PROCID MSGID STRUCTURED-DATA MSG

Example:
  <166>1 2018-06-27T12:17:46Z asa - - - %ASA-6-110002: Failed to locate egress interface...

---

## Part 2: Cisco ASA Log Format

### Core Message Structure

Every ASA event message follows this pattern, regardless of syslog envelope:

  %ASA-severity-message_id: message_text

| Field          | Description                                                    |
|----------------|----------------------------------------------------------------|
| %ASA           | Fixed literal prefix on all ASA messages                       |
| severity       | Single digit 0-7 matching syslog severity levels above         |
| message_id     | 6-digit numeric ID uniquely identifying the event type         |
| message_text   | Human-readable description with variable substitutions         |

### Full Log Line with Syslog Envelope

  <PRI> TIMESTAMP HOSTNAME : %ASA-severity-message_id: message_text

Real example (RFC 3164):
  <166>Jan 15 10:22:01 fw01 : %ASA-6-302013: Built outbound TCP connection 12345 for outside:203.0.113.1/443 (203.0.113.1/443) to inside:192.168.1.100/54321 (192.168.1.100/54321)

Real example (RFC 5424):
  <166>1 2024-01-15T10:22:01Z fw01 - - - %ASA-6-302013: Built outbound TCP connection...

### Message ID Numbering and Categories

The 6-digit message ID encodes the functional category in its first 3 digits:

| Range    | Category                                           |
|----------|----------------------------------------------------|
| 1xxxxx   | System, interfaces, failover, hardware             |
| 2xxxxx   | VPN failover, high availability                    |
| 3xxxxx   | Firewall, connection tracking, NAT                 |
| 4xxxxx   | Security, IPS, ACL, threat detection               |
| 5xxxxx   | User authentication, identity                      |
| 6xxxxx   | VPN crypto, IPSec, SSL, remote access              |
| 7xxxxx   | Application inspection, protocol handling          |
| 8xxxxx   | Management, platform, clustering                   |

### Common Variable Tokens

ASA message text uses descriptive named placeholders:

| Token            | Meaning                                        |
|------------------|------------------------------------------------|
| interface_name   | ASA interface (e.g., inside, outside)          |
| src_ip/port      | Source IP and port                             |
| dst_ip/port      | Destination IP and port                        |
| protocol         | IP protocol (TCP, UDP, ICMP, etc.)             |
| conn_id          | Unique connection tracking number              |
| direction        | inbound or outbound                            |
| reason           | Cause of event (e.g., SYN Timeout, FIN)        |
| username         | Authenticated user identity                    |
| group            | VPN tunnel group or policy name                |
| access_list      | Name of the ACL that matched                   |
| action           | permit or deny                                 |

### Parsing Regex

Extract the core ASA fields from any log line:

  %ASA-(?P<severity>\d)-(?P<message_id>\d{6}):\s+(?P<message_text>.+)

Full line including syslog envelope:

  ^(?P<timestamp>\S+\s+\S+\s+\S+|\S+T\S+Z)\s+(?P<hostname>\S+)\s+:\s+%ASA-(?P<severity>\d)-(?P<message_id>\d{6}):\s+(?P<message_text>.+)$

---

## Guidelines

- Always extract the message_id first -- it uniquely identifies the event type
- Check severity immediately: levels 1-3 indicate critical/error conditions requiring attention; 6-7 are routine
- Use the first 3 digits of the message_id to identify the functional area before looking up details
- ASA logs are interface-aware -- always note which interface (inside, outside, dmz) events occur on, as it provides essential network direction context
- Watch for paired messages: ASA often emits build/teardown pairs (e.g., 302013 "Built" + 302014 "Teardown") that together describe the full lifecycle of a connection
- For detailed per-message-ID definitions (message text templates, variable meanings, recommended actions), consult the reference files indexed below

---

## Reference Files

Detailed documentation for each message ID is maintained in separate files organized by range, matching Cisco's official documentation structure:

| File                                                     | Message ID Range  |
|----------------------------------------------------------|-------------------|
| `references/Syslog Messages 101001 to 199027.md`         | 101001 - 199027   |
| `references/Syslog Messages 201002 to 219002.md`         | 201002 - 219002   |
| `references/Syslog Messages 302003 to 342008.md`         | 302003 - 342008   |
| `references/Syslog Messages 400000 to 450002.md`         | 400000 - 450002   |
| `references/Syslog Messages 500001 to 520025.md`         | 500001 - 520025   |
| `references/Syslog Messages 602101 to 622102.md`         | 602101 - 622102   |
| `references/Syslog Messages 701001 to 714011.md`         | 701001 - 714011   |
| `references/Syslog Messages 715001 to 721019.md`         | 715001 - 721019   |
| `references/Syslog Messages 722001 to 776020.md`         | 722001 - 776020   |
| `references/Syslog Messages 776201 to 833333.md`         | 776201 - 833333   |

For serverity levels, refer to `references/Messages Listed by Severity Level.md`

For overall information of Cisco ASA syslog messages, refer to `references/About Cisco Secure Firewall ASA.md`


Official Cisco source: https://www.cisco.com/c/en/us/td/docs/security/asa/syslog/asa-syslog.html
