import { initiateDeveloperControlledWalletsClient } from '@circle-fin/developer-controlled-wallets';
import type { Wallet, Balance, Blockchain } from '@circle-fin/developer-controlled-wallets';
import { isValidEthereumAddress } from './utils';
import { getUSDCTokenId } from './chains';

export interface WalletConfig {
  apiKey: string;
  entitySecret: string;
  env?: 'sandbox' | 'production';
  defaultChain?: string;
  walletSetId?: string;
}


export class CircleWallet {
  private client: ReturnType<typeof initiateDeveloperControlledWalletsClient>;
  private config: WalletConfig;

  constructor(config: WalletConfig) {
    this.config = {
      env: 'sandbox',
      defaultChain: 'BASE-SEPOLIA',
      ...config
    };

    this.client = initiateDeveloperControlledWalletsClient({
      apiKey: this.config.apiKey,
      entitySecret: this.config.entitySecret,
    });
  }

  async createWalletSet(name: string): Promise<string> {
    const response = await this.client.createWalletSet({ name });
    if (!response.data?.walletSet?.id) {
      throw new Error('Failed to create wallet set');
    }
    return response.data.walletSet.id;
  }

  async createWallet(name: string, walletSetId?: string): Promise<Wallet> {
    const params: any = {
      blockchains: [this.config.defaultChain as Blockchain],
      count: 1,
      accountType: 'SCA',
    };

    if (walletSetId) {
      params.walletSetId = walletSetId;
    }

    const response = await this.client.createWallets(params);
    if (!response.data?.wallets || response.data.wallets.length === 0) {
      throw new Error('Failed to create wallet');
    }
    return response.data.wallets[0];
  }

  async getBalance(walletId: string): Promise<number> {
    const response = await this.client.getWalletTokenBalance({ id: walletId });
    if (!response.data?.tokenBalances) {
      return 0;
    }

    const usdcBalance = response.data.tokenBalances.find((b: Balance) =>
      b.token.symbol === 'USDC' ||
      b.token.symbol === 'USDC-TESTNET' ||
      (b.token.name && b.token.name.includes('USDC'))
    );

    return usdcBalance ? parseFloat(usdcBalance.amount) : 0;
  }

  async sendUSDC(params: {
    fromWalletId: string;
    toAddress: string;
    amount: string;
    tokenId?: string;
  }): Promise<{ transactionId: string; status: string }> {
    // Validate Ethereum address format
    if (!isValidEthereumAddress(params.toAddress)) {
      throw new Error('Invalid Ethereum address format. Address must be 0x followed by 40 hexadecimal characters');
    }

    const chain = this.config.defaultChain || 'ARC-TESTNET';
    const tokenId = params.tokenId || getUSDCTokenId(chain);

    if (!tokenId) {
      throw new Error(`USDC token ID not found for chain: ${chain}`);
    }

    const response = await this.client.createTransaction({
      walletId: params.fromWalletId,
      tokenId,
      destinationAddress: params.toAddress,
      amount: [params.amount],
      fee: { type: 'level', config: { feeLevel: 'MEDIUM' } }
    });

    if (!response.data?.id) {
      throw new Error('Transaction creation failed');
    }

    return {
      transactionId: response.data.id,
      status: response.data.state || 'INITIATED'
    };
  }

  async getTransactionStatus(transactionId: string): Promise<{
    state: string;
    txHash?: string;
    errorReason?: string;
  }> {
    const response = await this.client.getTransaction({ id: transactionId });
    const tx = response.data?.transaction;
    if (!tx) {
      throw new Error('Transaction not found');
    }
    return { state: tx.state, txHash: tx.txHash, errorReason: tx.errorReason };
  }

  async waitForTransaction(
    transactionId: string,
    maxWaitMs: number = 120000
  ): Promise<{ success: boolean; txHash?: string; error?: string }> {
    const startTime = Date.now();
    const pollInterval = 5000;

    while (Date.now() - startTime < maxWaitMs) {
      try {
        const status = await this.getTransactionStatus(transactionId);

        if (status.state === 'COMPLETE' || status.state === 'CONFIRMED') {
          return { success: true, txHash: status.txHash };
        }
        if (status.state === 'FAILED' || status.state === 'DENIED') {
          return { success: false, error: status.errorReason || 'Transaction failed' };
        }

        await new Promise(resolve => setTimeout(resolve, pollInterval));
      } catch {
        await new Promise(resolve => setTimeout(resolve, pollInterval));
      }
    }

    return { success: false, error: 'Transaction timeout' };
  }

  async listWallets(): Promise<Wallet[]> {
    const response = await this.client.listWallets({});
    return response.data?.wallets || [];
  }

  async getWallet(walletId: string): Promise<Wallet | null> {
    const wallets = await this.listWallets();
    return wallets.find(w => w.id === walletId) || null;
  }

  getUSDCTokenId(): string {
    const chain = this.config.defaultChain || 'ARC-TESTNET';
    return getUSDCTokenId(chain) || '';
  }

  async requestTestnetTokens(address: string, blockchain?: string): Promise<void> {
    const chain = blockchain || this.config.defaultChain || 'ARC-TESTNET';
    await this.client.requestTestnetTokens({
      address,
      blockchain: chain as any,
      usdc: true,
    });
  }
}
