/**
 * Utility functions for Circle Wallet Skill
 */

/**
 * Validates Ethereum address format
 * @param address - The address to validate
 * @returns true if valid Ethereum address format
 */
export function isValidEthereumAddress(address: string): boolean {
  return /^0x[a-fA-F0-9]{40}$/.test(address);
}

/**
 * Resolves a wallet identifier (ID or address) to a wallet ID
 * @param identifier - Wallet ID (UUID) or wallet address (0x...)
 * @param wallets - Array of wallets to search
 * @returns Wallet ID if found, null otherwise
 */
export function resolveWalletId(identifier: string, wallets: any[]): string | null {
  // Check if it's already a wallet ID (UUID format)
  if (/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i.test(identifier)) {
    return identifier;
  }

  // Check if it's a wallet address
  const wallet = wallets.find(w => w.address.toLowerCase() === identifier.toLowerCase());
  return wallet?.id || null;
}

/**
 * Validates USDC amount format and value
 * @param amount - Amount as string
 * @returns Validation result with error message or parsed value
 */
export function validateUSDCAmount(amount: string): { valid: boolean; error?: string; value?: number } {
  // Parse amount
  const amountNum = parseFloat(amount);

  // Check if valid number
  if (isNaN(amountNum)) {
    return { valid: false, error: 'Amount must be a valid number' };
  }

  // Check if positive
  if (amountNum <= 0) {
    return { valid: false, error: 'Amount must be greater than 0' };
  }

  // Check decimal places (USDC has 6 decimals max)
  const decimalMatch = amount.match(/\.(\d+)/);
  if (decimalMatch && decimalMatch[1].length > 6) {
    return { valid: false, error: 'Amount cannot have more than 6 decimal places (USDC standard)' };
  }

  return { valid: true, value: amountNum };
}

/**
 * Formats USDC balance for display (removes trailing zeros)
 * @param balance - Balance as number or string
 * @returns Formatted balance string
 */
export function formatUSDCBalance(balance: number | string): string {
  const balanceNum = typeof balance === 'number' ? balance : parseFloat(balance);
  return balanceNum.toFixed(6).replace(/\.?0+$/, '');
}
