/**
 * Entity Secret Management
 * Uses Circle SDK's functions for entity secret generation and registration
 */

import * as crypto from 'crypto';
import { registerEntitySecretCiphertext } from '@circle-fin/developer-controlled-wallets';

/**
 * Generate a new entity secret (32 bytes hex string)
 */
export function generateEntitySecret(): string {
  const buffer = crypto.randomBytes(32);
  return buffer.toString('hex');
}

/**
 * Register entity secret with Circle
 * Uses Circle SDK's registerEntitySecretCiphertext
 */
export async function registerEntitySecret(
  apiKey: string,
  entitySecret: string
): Promise<{ success: boolean; error?: string }> {
  try {
    // Register the entity secret with Circle using SDK
    // The SDK handles generating the ciphertext internally
    const response = await registerEntitySecretCiphertext({
      entitySecret,
      apiKey
    });

    if (response.data) {
      return { success: true };
    }

    return {
      success: false,
      error: 'Registration failed - no data returned'
    };
  } catch (error) {
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error'
    };
  }
}
