/**
 * Configuration Management
 */

import * as fs from 'fs';
import * as path from 'path';
import type { WalletConfig } from './wallet';

const CONFIG_DIR = path.join(process.env.HOME || '~', '.openclaw', 'circle-wallet');
const CONFIG_FILE = path.join(CONFIG_DIR, 'config.json');

export function ensureConfigDir(): void {
  if (!fs.existsSync(CONFIG_DIR)) {
    fs.mkdirSync(CONFIG_DIR, { recursive: true });
  }
}

export function loadConfig(): WalletConfig {
  ensureConfigDir();

  if (!fs.existsSync(CONFIG_FILE)) {
    throw new Error('No configuration found. Run "circle-wallet setup" first.');
  }

  const config = JSON.parse(fs.readFileSync(CONFIG_FILE, 'utf-8'));
  return config;
}

export function saveConfig(config: WalletConfig): void {
  ensureConfigDir();
  fs.writeFileSync(CONFIG_FILE, JSON.stringify(config, null, 2));
}

export function configExists(): boolean {
  return fs.existsSync(CONFIG_FILE);
}
