#!/usr/bin/env node
/**
 * Circle Wallet Skill CLI
 * Command-line interface for OpenClaw agents
 */

import { Command } from 'commander';
import { CircleWallet } from './wallet';
import { loadConfig, saveConfig, ensureConfigDir } from './config';
import { generateEntitySecret, registerEntitySecret } from './entity';
import { isValidEthereumAddress, resolveWalletId, validateUSDCAmount, formatUSDCBalance } from './utils';
import { SUPPORTED_CHAINS, getMainnetChains, getTestnetChains, getChainInfo, isValidChain } from './chains';
import * as fs from 'fs';
import * as path from 'path';

const program = new Command();

program
  .name('circle-wallet')
  .description('Circle Developer-Controlled Wallets for OpenClaw agents')
  .version('1.0.0');

/**
 * Setup command - Initialize configuration
 */
program
  .command('setup')
  .description('Generate and configure Entity Secret')
  .requiredOption('--api-key <key>', 'Circle API key')
  .option('--env <environment>', 'Environment (sandbox or production)', 'sandbox')
  .action(async (options: { apiKey: string; env: string }) => {
    console.log('Circle Wallet Setup\n');

    ensureConfigDir();

    const apiKey = options.apiKey;
    const env = options.env as 'sandbox' | 'production';

    if (env !== 'sandbox' && env !== 'production') {
      console.error('❌ Environment must be "sandbox" or "production"');
      process.exit(1);
    }

    console.log(`Environment: ${env}`);

    // Generate entity secret
    console.log('\nGenerating entity secret...');
    const entitySecret = generateEntitySecret();

    // Register entity secret with Circle
    console.log('Registering entity secret with Circle...');
    const result = await registerEntitySecret(apiKey, entitySecret);

    if (!result.success) {
      console.error(`❌ Registration failed: ${result.error}`);
      console.log('\nPlease check:');
      console.log('  1. Your API key is valid');
      console.log('  2. You have not already registered an entity secret');
      console.log('\nAlready have credentials?');
      console.log('  circle-wallet configure --api-key <key> --entity-secret <secret>');
      process.exit(1);
    }

    // Clear old wallets when setting up new entity secret
    const walletsFile = path.join(process.env.HOME || '~', '.openclaw', 'circle-wallet', 'wallets.json');
    const defaultWalletFile = path.join(process.env.HOME || '~', '.openclaw', 'circle-wallet', 'default-wallet.txt');
    if (fs.existsSync(walletsFile)) {
      fs.unlinkSync(walletsFile);
    }
    if (fs.existsSync(defaultWalletFile)) {
      fs.unlinkSync(defaultWalletFile);
    }

    // Save configuration
    saveConfig({
      apiKey,
      entitySecret,
      env,
      defaultChain: env === 'sandbox' ? 'ARC-TESTNET' : 'BASE'
    });

    console.log('\n✅ Setup complete!');
    console.log('   Entity secret generated and registered');
    console.log('   Configuration saved to ~/.openclaw/circle-wallet/');
    console.log('\nGetting Started Guide:');
    console.log('  1. circle-wallet create           # Create your first wallet');
    console.log('  2. circle-wallet drip             # Get 20 testnet USDC (sandbox only)');
    console.log('  3. circle-wallet balance          # Check your balance');
    console.log('  4. circle-wallet create "Wallet2" # Create a second wallet (optional)');
    console.log('  5. circle-wallet send <address> <amount> --from <wallet-address>');
  });

/**
 * Configure command - Manual configuration with existing credentials
 */
program
  .command('configure')
  .description('Configure with existing credentials')
  .requiredOption('--api-key <key>', 'Circle API key')
  .requiredOption('--entity-secret <secret>', 'Entity secret')
  .option('--env <environment>', 'Environment (sandbox or production)', 'sandbox')
  .action(async (options: { apiKey: string; entitySecret: string; env: string }) => {
    console.log('Circle Wallet Configuration\n');

    ensureConfigDir();

    const env = options.env as 'sandbox' | 'production';

    if (env !== 'sandbox' && env !== 'production') {
      console.error('❌ Environment must be "sandbox" or "production"');
      process.exit(1);
    }

    // Clear old wallets when configuring new credentials
    const walletsFile = path.join(process.env.HOME || '~', '.openclaw', 'circle-wallet', 'wallets.json');
    const defaultWalletFile = path.join(process.env.HOME || '~', '.openclaw', 'circle-wallet', 'default-wallet.txt');
    if (fs.existsSync(walletsFile)) {
      fs.unlinkSync(walletsFile);
    }
    if (fs.existsSync(defaultWalletFile)) {
      fs.unlinkSync(defaultWalletFile);
    }

    // Save configuration
    saveConfig({
      apiKey: options.apiKey,
      entitySecret: options.entitySecret,
      env,
      defaultChain: env === 'sandbox' ? 'ARC-TESTNET' : 'BASE'
    });

    console.log('✅ Configuration saved!');
    console.log(`   Environment: ${env}`);
    console.log(`   Default Chain: ${env === 'sandbox' ? 'ARC-TESTNET' : 'BASE'}`);
    console.log('   Configuration saved to ~/.openclaw/circle-wallet/');
    console.log('\nGetting Started Guide:');
    console.log('  1. circle-wallet create           # Create your first wallet');
    console.log('  2. circle-wallet drip             # Get 20 testnet USDC (sandbox only)');
    console.log('  3. circle-wallet balance          # Check your balance');
    console.log('  4. circle-wallet create "Wallet2" # Create a second wallet (optional)');
    console.log('  5. circle-wallet send <address> <amount> --from <wallet-address>');
  });

/**
 * Create wallet command
 */
program
  .command('create')
  .description('Create a new wallet')
  .argument('[name]', 'Wallet name', 'Default Wallet')
  .option('--chain <blockchain>', 'Blockchain to create wallet on (e.g., BASE-SEPOLIA, ETH, MATIC)')
  .action(async (name: string, options: { chain?: string }) => {
    try {
      const config = loadConfig();

      // Handle chain parameter
      let chain = options.chain ? options.chain.toUpperCase() : config.defaultChain || 'ARC-TESTNET';

      // Validate chain
      if (!isValidChain(chain)) {
        console.error(`❌ Invalid chain: ${chain}`);
        console.log('Run "circle-wallet chains" to see supported blockchains');
        process.exit(1);
      }

      // Create wallet with specified chain
      const walletConfig = { ...config, defaultChain: chain };
      const wallet = new CircleWallet(walletConfig);

      // Check if wallet set exists in config
      let walletSetId = (config as any).walletSetId;

      if (!walletSetId) {
        console.log('📦 Creating wallet set...');
        walletSetId = await wallet.createWalletSet('OpenClaw Wallet Set');

        // Save wallet set ID to config
        const updatedConfig = { ...config, walletSetId };
        saveConfig(updatedConfig);
        console.log('   Wallet Set ID:', walletSetId);
      }

      console.log(`🔨 Creating wallet: ${name}...`);
      const newWallet = await wallet.createWallet(name, walletSetId);

      console.log('\n✅ Wallet created!');
      console.log(`   ID: ${newWallet.id}`);
      console.log(`   Address: ${newWallet.address}`);
      console.log(`   Chain: ${newWallet.blockchain}`);

      // Save as default if first wallet
      const wallets = loadWallets();
      const walletCount = wallets.length;

      if (walletCount === 0) {
        saveDefaultWallet(newWallet.id);
      }
      saveWallet(newWallet);

      // Add contextual next steps
      if (walletCount === 0) {
        // First wallet
        console.log('\nNext Steps:');
        if (config.env === 'sandbox') {
          console.log('  1. circle-wallet drip                    # Get 20 testnet USDC');
        } else {
          console.log('  1. Send USDC to your wallet address      # Fund your wallet');
        }
        console.log('  2. circle-wallet balance                 # Check balance');
        console.log('  3. circle-wallet create "Second Wallet"  # Create another wallet');
      } else {
        // Subsequent wallets
        console.log('\nTips:');
        console.log(`  - You now have ${walletCount + 1} wallet${walletCount > 0 ? 's' : ''}`);
        console.log('  - Run "circle-wallet list" to see all wallets');
        console.log('  - Send between wallets: circle-wallet send <to-address> <amount> --from <from-address>');
      }

    } catch (error) {
      console.error('❌ Error:', error);
      process.exit(1);
    }
  });

/**
 * Balance command
 */
program
  .command('balance')
  .description('Check USDC balance')
  .argument('[wallet-id]', 'Wallet ID (uses default if not provided)')
  .action(async (walletId?: string) => {
    try {
      const config = loadConfig();
      const wallet = new CircleWallet(config);

      const targetWalletId = walletId || getDefaultWallet();
      if (!targetWalletId) {
        console.error('❌ No wallet ID provided and no default wallet set');
        console.log('Run "circle-wallet:create" first');
        process.exit(1);
      }

      console.log('🔍 Checking balance...');
      const balance = await wallet.getBalance(targetWalletId);

      // Display balance with proper decimal precision (up to 6 decimals for USDC)
      const formattedBalance = formatUSDCBalance(balance);
      console.log(`\n💰 Balance: ${formattedBalance} USDC`);

    } catch (error) {
      console.error('❌ Error:', error);
      process.exit(1);
    }
  });

/**
 * Send command
 */
program
  .command('send')
  .description('Send USDC to an address')
  .argument('<to-address>', 'Recipient address')
  .argument('<amount>', 'Amount in USDC')
  .option('-f, --from <wallet-identifier>', 'From wallet (ID or address) - uses default if not provided')
  .action(async (toAddress: string, amount: string, options: { from?: string }) => {
    try {
      const config = loadConfig();
      const walletClient = new CircleWallet(config);

      // Validate recipient address format
      if (!isValidEthereumAddress(toAddress)) {
        console.error('❌ Invalid Ethereum address format');
        console.log('Address must be 0x followed by 40 hexadecimal characters');
        process.exit(1);
      }

      // Resolve from wallet identifier (ID or address)
      let fromWalletId: string | null = null;

      if (options.from) {
        const wallets = await walletClient.listWallets();
        fromWalletId = resolveWalletId(options.from, wallets);

        if (!fromWalletId) {
          console.error('❌ Invalid wallet identifier');
          console.log('Use wallet ID or address. Run "circle-wallet list" to see your wallets.');
          process.exit(1);
        }
      } else {
        fromWalletId = getDefaultWallet();
      }

      if (!fromWalletId) {
        console.error('❌ No wallet configured');
        console.log('Run "circle-wallet create" first');
        process.exit(1);
      }

      // Get wallet info to determine chain
      const wallet = await walletClient.getWallet(fromWalletId);
      if (!wallet) {
        console.error('❌ Wallet not found');
        process.exit(1);
      }

      // Validate amount format
      const validation = validateUSDCAmount(amount);
      if (!validation.valid) {
        console.error(`❌ ${validation.error}`);
        process.exit(1);
      }
      const amountNum = validation.value!;

      // Check balance before sending
      console.log('Checking balance...');
      const balance = await walletClient.getBalance(fromWalletId);

      if (balance < amountNum) {
        console.error(`❌ Insufficient balance: ${balance} USDC < ${amountNum} USDC`);
        process.exit(1);
      }

      console.log(`Sending ${amount} USDC to ${toAddress}...`);
      console.log(`Chain: ${wallet.blockchain}`);

      // Use chain-specific config for sending
      const sendConfig = { ...config, defaultChain: wallet.blockchain };
      const chainWallet = new CircleWallet(sendConfig);

      const result = await chainWallet.sendUSDC({
        fromWalletId,
        toAddress,
        amount
      });

      console.log(`\n✅ Transaction created!`);
      console.log(`   ID: ${result.transactionId}`);
      console.log(`   Status: ${result.status}`);

      // Wait for confirmation
      console.log('\nWaiting for confirmation...');
      const final = await chainWallet.waitForTransaction(result.transactionId);

      if (final.success) {
        console.log(`✅ Transaction complete!`);
        console.log(`   TX Hash: ${final.txHash}`);
      } else {
        console.error(`❌ Transaction failed: ${final.error}`);
      }

    } catch (error) {
      console.error('❌ Error:', error);
      process.exit(1);
    }
  });

/**
 * List wallets command
 */
program
  .command('list')
  .description('List all wallets with balances')
  .action(async () => {
    try {
      const config = loadConfig();
      const wallet = new CircleWallet(config);

      console.log('Your Wallets:\n');
      const wallets = await wallet.listWallets();

      if (wallets.length === 0) {
        console.log('No wallets found. Run "circle-wallet create" to create one.');
        return;
      }

      const defaultWalletId = getDefaultWallet();

      for (const w of wallets) {
        const isDefault = w.id === defaultWalletId ? ' (default)' : '';

        // Fetch balance for each wallet
        let balanceDisplay = 'Loading...';
        try {
          const balance = await wallet.getBalance(w.id);
          const formattedBalance = formatUSDCBalance(balance);
          balanceDisplay = `${formattedBalance} USDC`;
        } catch (error) {
          balanceDisplay = 'Error fetching balance';
        }

        console.log(`${w.address}${isDefault}`);
        console.log(`  ID: ${w.id}`);
        console.log(`  Chain: ${w.blockchain}`);
        console.log(`  State: ${w.state}`);
        console.log(`  Balance: ${balanceDisplay}\n`);
      }

    } catch (error) {
      console.error('❌ Error:', error);
      process.exit(1);
    }
  });

/**
 * Config command
 */
program
  .command('config')
  .description('View current configuration')
  .action(() => {
    try {
      const config = loadConfig();

      console.log('⚙️  Circle Wallet Configuration:\n');
      console.log(`  API Key: ${config.apiKey.substring(0, 20)}...`);
      console.log(`  Environment: ${config.env}`);
      console.log(`  Default Chain: ${config.defaultChain}`);
      console.log(`  Default Wallet: ${getDefaultWallet() || 'None'}`);

    } catch (error) {
      console.error('❌ No configuration found. Run "circle-wallet:setup" first.');
      process.exit(1);
    }
  });

/**
 * Drip command (testnet only)
 */
program
  .command('drip')
  .description('Request testnet USDC from faucet (sandbox only)')
  .argument('[address]', 'Address to fund (uses default wallet if not provided)')
  .option('--chain <blockchain>', 'Blockchain to request tokens on (e.g., ARC-TESTNET, BASE-SEPOLIA)')
  .action(async (address?: string, options?: { chain?: string }) => {
    const config = loadConfig();

    if (config.env !== 'sandbox') {
      console.error('❌ Faucet only available in sandbox environment');
      process.exit(1);
    }

    // Determine which chain to use for drip
    let targetChain = config.defaultChain || 'ARC-TESTNET';

    if (options?.chain) {
      // User specified chain explicitly
      const chainUpper = options.chain.toUpperCase();

      // Validate chain exists
      if (!isValidChain(chainUpper)) {
        console.error(`❌ Invalid chain: ${chainUpper}`);
        console.log('Run "circle-wallet chains --testnet" to see supported testnet chains');
        process.exit(1);
      }

      // Verify it's a testnet chain
      const chainInfo = getChainInfo(chainUpper);
      if (chainInfo?.network !== 'testnet') {
        console.error(`❌ Chain ${chainUpper} is not a testnet. Faucet only works on testnets.`);
        process.exit(1);
      }

      targetChain = chainUpper;
    }

    let targetAddress = address;
    if (!targetAddress) {
      // Get default wallet address from local storage first
      const walletId = getDefaultWallet();
      if (walletId) {
        // Try local wallets.json first
        const localWallets = loadWallets();
        let defaultWallet = localWallets.find(w => w.id === walletId);

        // If not found locally, query API
        if (!defaultWallet) {
          const wallet = new CircleWallet(config);
          const wallets = await wallet.listWallets();
          defaultWallet = wallets.find(w => w.id === walletId);
        }

        if (defaultWallet) {
          targetAddress = defaultWallet.address;
          // Auto-detect chain from wallet if no explicit chain specified
          if (!options?.chain) {
            targetChain = defaultWallet.blockchain;
            console.log(`🔍 Auto-detected chain from default wallet: ${targetChain}`);
          }
        }
      }
    }

    if (!targetAddress) {
      console.error('❌ No address provided and no default wallet found');
      console.log('Run "circle-wallet create" to create your first wallet');
      process.exit(1);
    }

    // Validate address format
    if (!isValidEthereumAddress(targetAddress)) {
      console.error('❌ Invalid Ethereum address format');
      console.log('Address must be 0x followed by 40 hexadecimal characters');
      console.log('Run "circle-wallet list" to see your wallet addresses');
      process.exit(1);
    }

    console.log('💧 Requesting testnet USDC...');
    console.log(`   Address: ${targetAddress}`);
    console.log(`   Chain: ${targetChain}`);

    try {
      const wallet = new CircleWallet(config);
      await wallet.requestTestnetTokens(targetAddress, targetChain);

      console.log('\n✅ Testnet USDC requested successfully!');
      console.log('Tokens should arrive in a few moments');
      console.log('Gas fees covered by Circle Gas Station');
      console.log('\nNext Steps:');
      console.log('  1. circle-wallet balance                 # Check your balance');
      console.log('  2. circle-wallet create "Second Wallet"  # Create another wallet');
      console.log('  3. circle-wallet send <to-address> 5 --from <from-address>  # Send USDC');
    } catch (error: any) {
      console.error('\n❌ Faucet request failed:', error.message || error);
      console.log('\nAlternative: Visit https://faucet.circle.com');
      console.log('Note: Faucet drips 20 USDC every 2 hours');
    }
  });

// Helper functions
function loadWallets(): any[] {
  const walletsFile = path.join(process.env.HOME || '~', '.openclaw', 'circle-wallet', 'wallets.json');
  if (!fs.existsSync(walletsFile)) {
    return [];
  }
  return JSON.parse(fs.readFileSync(walletsFile, 'utf-8'));
}

function saveWallet(wallet: any): void {
  const walletsFile = path.join(process.env.HOME || '~', '.openclaw', 'circle-wallet', 'wallets.json');
  const wallets = loadWallets();
  wallets.push(wallet);
  fs.writeFileSync(walletsFile, JSON.stringify(wallets, null, 2));
}

function getDefaultWallet(): string | null {
  const configFile = path.join(process.env.HOME || '~', '.openclaw', 'circle-wallet', 'default-wallet.txt');
  if (!fs.existsSync(configFile)) {
    return null;
  }
  return fs.readFileSync(configFile, 'utf-8').trim();
}

function saveDefaultWallet(walletId: string): void {
  const configFile = path.join(process.env.HOME || '~', '.openclaw', 'circle-wallet', 'default-wallet.txt');
  fs.writeFileSync(configFile, walletId);
}

/**
 * Chains command - List supported blockchains
 */
program
  .command('chains')
  .description('List all supported blockchains')
  .option('--show-tokens', 'Show USDC token IDs')
  .option('--mainnet', 'Show only mainnet chains')
  .option('--testnet', 'Show only testnet chains')
  .action((options: { showTokens?: boolean; mainnet?: boolean; testnet?: boolean }) => {
    console.log('Supported Blockchains\n');

    let chains = Object.values(SUPPORTED_CHAINS);

    if (options.mainnet) {
      chains = getMainnetChains();
    } else if (options.testnet) {
      chains = getTestnetChains();
    }

    // Group by network
    const mainnets = chains.filter(c => c.network === 'mainnet');
    const testnets = chains.filter(c => c.network === 'testnet');

    if (mainnets.length > 0 && !options.testnet) {
      console.log('MAINNETS:');
      mainnets.forEach(chain => {
        const chainId = chain.id.padEnd(20);
        console.log(`  ${chainId} ${chain.name}`);
        if (options.showTokens && chain.usdcTokenId) {
          console.log(`    Token ID: ${chain.usdcTokenId}`);
        }
      });
      console.log('');
    }

    if (testnets.length > 0 && !options.mainnet) {
      console.log('TESTNETS:');
      testnets.forEach(chain => {
        const chainId = chain.id.padEnd(20);
        console.log(`  ${chainId} ${chain.name}`);
        if (options.showTokens && chain.usdcTokenId) {
          console.log(`    Token ID: ${chain.usdcTokenId}`);
        }
      });
      console.log('');
    }

    console.log(`Total: ${chains.length} chains (${mainnets.length} mainnet, ${testnets.length} testnet)`);
  });

program.parse();
