/**
 * Supported Blockchains and USDC Token IDs
 * Reference: https://developers.circle.com/wallets/monitored-tokens
 */

export interface ChainInfo {
  id: string;           // Circle SDK blockchain ID
  name: string;         // Human-readable name
  network: 'mainnet' | 'testnet';
  usdcTokenId: string;  // USDC token ID for this chain
}

export const SUPPORTED_CHAINS: Record<string, ChainInfo> = {
  // Mainnets
  'APTOS': {
    id: 'APTOS',
    name: 'Aptos Mainnet',
    network: 'mainnet',
    usdcTokenId: '298eebe2-3131-5183-b528-f925f70848d0'
  },
  'ARB': {
    id: 'ARB',
    name: 'Arbitrum One',
    network: 'mainnet',
    usdcTokenId: 'c87ffcb4-e2cf-5e67-84c6-388c965d2a66'
  },
  'AVAX': {
    id: 'AVAX',
    name: 'Avalanche C-Chain',
    network: 'mainnet',
    usdcTokenId: '7efdfdbf-1799-5089-a588-31beb97ba755'
  },
  'BASE': {
    id: 'BASE',
    name: 'Base',
    network: 'mainnet',
    usdcTokenId: 'aa7bb533-aeb8-535c-bd65-354aed91ea3d'
  },
  'ETH': {
    id: 'ETH',
    name: 'Ethereum Mainnet',
    network: 'mainnet',
    usdcTokenId: 'b037d751-fb22-5f0d-bae6-47373e7ae3e3'
  },
  'MONAD': {
    id: 'MONAD',
    name: 'Monad Mainnet',
    network: 'mainnet',
    usdcTokenId: '95401815-48ec-59eb-a06c-8fd1a8e82a8f'
  },
  'OP': {
    id: 'OP',
    name: 'Optimism',
    network: 'mainnet',
    usdcTokenId: '' // Mainnet token ID not yet available
  },
  'MATIC': {
    id: 'MATIC',
    name: 'Polygon PoS',
    network: 'mainnet',
    usdcTokenId: 'db6905b9-8bcd-5537-8b08-f5548bdf7925'
  },
  'SOL': {
    id: 'SOL',
    name: 'Solana Mainnet',
    network: 'mainnet',
    usdcTokenId: '33ca4ef6-2500-5d79-82bf-e3036139cc29'
  },
  'UNI': {
    id: 'UNI',
    name: 'Unichain',
    network: 'mainnet',
    usdcTokenId: '643096e8-e4c5-5ee7-a5d5-38c80fdc3f3b'
  },

  // Testnets
  'APTOS-TESTNET': {
    id: 'APTOS-TESTNET',
    name: 'Aptos Testnet',
    network: 'testnet',
    usdcTokenId: 'e3cbdafc-42c3-58cc-ae4c-b31dbb10354c'
  },
  'ARB-SEPOLIA': {
    id: 'ARB-SEPOLIA',
    name: 'Arbitrum Sepolia',
    network: 'testnet',
    usdcTokenId: '4b8daacc-5f47-5909-a3ba-30d171ebad98'
  },
  'ARC-TESTNET': {
    id: 'ARC-TESTNET',
    name: 'Arc Testnet',
    network: 'testnet',
    usdcTokenId: '15dc2b5d-0994-58b0-bf8c-3a0501148ee8'
  },
  'AVAX-FUJI': {
    id: 'AVAX-FUJI',
    name: 'Avalanche Fuji',
    network: 'testnet',
    usdcTokenId: 'ff47a560-9795-5b7c-adfc-8f47dad9e06a'
  },
  'BASE-SEPOLIA': {
    id: 'BASE-SEPOLIA',
    name: 'Base Sepolia',
    network: 'testnet',
    usdcTokenId: 'bdf128b4-827b-5267-8f9e-243694989b5f'
  },
  'ETH-SEPOLIA': {
    id: 'ETH-SEPOLIA',
    name: 'Ethereum Sepolia',
    network: 'testnet',
    usdcTokenId: '5797fbd6-3795-519d-84ca-ec4c5f80c3b1'
  },
  'MONAD-TESTNET': {
    id: 'MONAD-TESTNET',
    name: 'Monad Testnet',
    network: 'testnet',
    usdcTokenId: '17fe0589-b511-546c-9eff-84087524f47e'
  },
  'OP-SEPOLIA': {
    id: 'OP-SEPOLIA',
    name: 'Optimism Sepolia',
    network: 'testnet',
    usdcTokenId: '20749a68-e55b-57d3-8359-195543efc2d4'
  },
  'MATIC-AMOY': {
    id: 'MATIC-AMOY',
    name: 'Polygon Amoy',
    network: 'testnet',
    usdcTokenId: '36b6931a-873a-56a8-8a27-b706b17104ee'
  },
  'SOL-DEVNET': {
    id: 'SOL-DEVNET',
    name: 'Solana Devnet',
    network: 'testnet',
    usdcTokenId: '8fb3cadb-0ef4-573d-8fcd-e194f961c728'
  },
  'UNI-SEPOLIA': {
    id: 'UNI-SEPOLIA',
    name: 'Unichain Sepolia',
    network: 'testnet',
    usdcTokenId: '13ef30cd-309b-5c41-98cc-0fd68c4c8c44'
  },
};

/**
 * Get chain info by ID
 */
export function getChainInfo(chainId: string): ChainInfo | undefined {
  return SUPPORTED_CHAINS[chainId.toUpperCase()];
}

/**
 * Get all mainnet chains
 */
export function getMainnetChains(): ChainInfo[] {
  return Object.values(SUPPORTED_CHAINS).filter(c => c.network === 'mainnet');
}

/**
 * Get all testnet chains
 */
export function getTestnetChains(): ChainInfo[] {
  return Object.values(SUPPORTED_CHAINS).filter(c => c.network === 'testnet');
}

/**
 * Validate if chain ID is supported
 */
export function isValidChain(chainId: string): boolean {
  return chainId.toUpperCase() in SUPPORTED_CHAINS;
}

/**
 * Get USDC token ID for a chain
 */
export function getUSDCTokenId(chainId: string): string | undefined {
  const chain = getChainInfo(chainId);
  return chain?.usdcTokenId || undefined;
}
