---
name: chargebee
description: |
  Chargebee API integration with managed OAuth. Manage subscriptions, customers, invoices, and hosted checkout pages. Use this skill when users want to interact with Chargebee billing data. For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
compatibility: Requires network access and valid Maton API key
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    requires:
      env:
        - MATON_API_KEY
---

# Chargebee

Access the Chargebee API with managed OAuth authentication. Manage subscriptions, customers, invoices, and billing workflows.

## Quick Start

```bash
# List customers
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/chargebee/api/v2/customers?limit=10')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/chargebee/{native-api-path}
```

Replace `{native-api-path}` with the actual Chargebee API endpoint path. The gateway proxies requests to `{subdomain}.chargebee.com` (automatically replaced with your connection config) and automatically injects authentication.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Chargebee connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=chargebee&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'chargebee'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "21fd90f9-5935-43cd-b6c8-bde9d915ca80",
    "status": "ACTIVE",
    "creation_time": "2025-12-08T07:20:53.488460Z",
    "last_updated_time": "2026-01-31T20:03:32.593153Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "chargebee",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple Chargebee connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/chargebee/api/v2/customers')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Maton-Connection', '21fd90f9-5935-43cd-b6c8-bde9d915ca80')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

### Customers

#### List Customers

```bash
GET /chargebee/api/v2/customers?limit=10
```

#### Get Customer

```bash
GET /chargebee/api/v2/customers/{customerId}
```

#### Create Customer

```bash
POST /chargebee/api/v2/customers
Content-Type: application/x-www-form-urlencoded

first_name=John&last_name=Doe&email=john@example.com
```

#### Update Customer

```bash
POST /chargebee/api/v2/customers/{customerId}
Content-Type: application/x-www-form-urlencoded

first_name=Jane
```

### Subscriptions

#### List Subscriptions

```bash
GET /chargebee/api/v2/subscriptions?limit=10
```

#### Get Subscription

```bash
GET /chargebee/api/v2/subscriptions/{subscriptionId}
```

#### Create Subscription

```bash
POST /chargebee/api/v2/subscriptions
Content-Type: application/x-www-form-urlencoded

plan_id=basic-plan&customer[email]=john@example.com&customer[first_name]=John
```

#### Cancel Subscription

```bash
POST /chargebee/api/v2/subscriptions/{subscriptionId}/cancel
Content-Type: application/x-www-form-urlencoded

end_of_term=true
```

### Item Prices (Product Catalog 2.0)

#### List Item Prices

```bash
GET /chargebee/api/v2/item_prices?limit=10
```

### Items

#### List Items

```bash
GET /chargebee/api/v2/items?limit=10
```

### Invoices

#### List Invoices

```bash
GET /chargebee/api/v2/invoices?limit=10
```

#### Download Invoice PDF

```bash
POST /chargebee/api/v2/invoices/{invoiceId}/pdf
```

### Hosted Pages

#### Checkout New Subscription

```bash
POST /chargebee/api/v2/hosted_pages/checkout_new_for_items
Content-Type: application/x-www-form-urlencoded

subscription[plan_id]=basic-plan&customer[email]=john@example.com
```

### Portal Sessions

#### Create Portal Session

```bash
POST /chargebee/api/v2/portal_sessions
Content-Type: application/x-www-form-urlencoded

customer[id]=cust_123
```

## Filtering

```bash
GET /chargebee/api/v2/subscriptions?status[is]=active
GET /chargebee/api/v2/customers?email[is]=john@example.com
GET /chargebee/api/v2/invoices?date[after]=1704067200
```

## Code Examples

### JavaScript

```javascript
const response = await fetch(
  'https://gateway.maton.ai/chargebee/api/v2/customers?limit=10',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);
```

### Python

```python
import os
import requests

response = requests.get(
    'https://gateway.maton.ai/chargebee/api/v2/customers',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'},
    params={'limit': 10}
)
```

## Notes

- Uses form-urlencoded data for POST requests
- Nested objects use bracket notation: `customer[email]`
- Timestamps are Unix timestamps
- List responses include `next_offset` for pagination
- Product Catalog 2.0: use `item_prices` and `items`
- IMPORTANT: When using curl commands, use `curl -g` when URLs contain brackets (`fields[]`, `sort[]`, `records[]`) to disable glob parsing
- IMPORTANT: When piping curl output to `jq` or other commands, environment variables like `$MATON_API_KEY` may not expand correctly in some shell environments. You may get "Invalid API key" errors when piping.

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Missing Chargebee connection |
| 401 | Invalid or missing Maton API key |
| 429 | Rate limited (10 req/sec per account) |
| 4xx/5xx | Passthrough error from Chargebee API |

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `chargebee`. For example:

- Correct: `https://gateway.maton.ai/chargebee/api/v2/customers`
- Incorrect: `https://gateway.maton.ai/api/v2/customers`

## Resources

- [Chargebee API Overview](https://apidocs.chargebee.com/docs/api)
- [Customers](https://apidocs.chargebee.com/docs/api/customers)
- [Subscriptions](https://apidocs.chargebee.com/docs/api/subscriptions)
- [Invoices](https://apidocs.chargebee.com/docs/api/invoices)
- [Hosted Pages](https://apidocs.chargebee.com/docs/api/hosted_pages)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
