# Cetus Protocol SDK v2 - OpenClaw Integration Guide

This guide covers all Cetus SDK v2 packages for building DeFi applications on Sui.

---

## Table of Contents

1. [Common SDK](#common-sdk)
2. [CLMM SDK](#clmm-sdk)
3. [DLMM SDK](#dlmm-sdk)
4. [Vaults SDK](#vaults-sdk)
5. [Farms SDK](#farms-sdk)
6. [xCETUS SDK](#xcetus-sdk)
7. [Limit Order SDK](#limit-order-sdk)
8. [Burn SDK](#burn-sdk)
9. [DCA SDK](#dca-sdk)
10. [Zap SDK](#zap-sdk)
11. [Aggregator SDK](#aggregator-sdk)

---

## Common SDK

Foundational utility library shared across all Cetus SDKs.

```bash
npm install @cetusprotocol/common-sdk
```

Provides essential utilities and shared functionality for protocol interactions. All other SDKs depend on this package.

---

## CLMM SDK

Concentrated Liquidity Market Maker - the core AMM of Cetus.

```bash
npm install @cetusprotocol/sui-clmm-sdk
```

### Initialization

```typescript
import { CetusClmmSDK } from '@cetusprotocol/sui-clmm-sdk'

// Default mainnet
const sdk = CetusClmmSDK.createSDK()

// Custom environment
const sdk = CetusClmmSDK.createSDK({ env: 'mainnet' }) // or 'testnet'

// Custom RPC
const sdk = CetusClmmSDK.createSDK({ env: 'mainnet', full_rpc_url: 'YOUR_URL' })

// Custom SuiClient
const sdk = CetusClmmSDK.createSDK({ env: 'mainnet', sui_client: yourClient })

// Set sender address (required before tx operations)
sdk.setSenderAddress('YOUR_SUI_ADDRESS')
```

---

## DLMM SDK

Dynamic Liquidity Market Maker - discrete bin-based AMM with dynamic fees.

```bash
npm install @cetusprotocol/dlmm-sdk
```

### Initialization

```typescript
import { CetusDlmmSDK } from '@cetusprotocol/dlmm-sdk'

const sdk = CetusDlmmSDK.createSDK()
// or with options: CetusDlmmSDK.createSDK({ env, full_rpc_url, sui_client })
sdk.setSenderAddress(walletAddress)
```

### Pool Operations

```typescript
// Get all pools
const pools = await sdk.Dlmm.getPoolList()

// Get specific pool
const pool = await sdk.Dlmm.getPool(pool_id)

// Get bin configuration
const binConfig = await sdk.Dlmm.getBinConfig(config_id)

// Get pool transaction history
const history = await sdk.Dlmm.getPoolTransactionHistory(pool_id)
```

### Position Management

Three liquidity distribution strategies:
- **Spot** - even distribution across bins
- **BidAsk** - concentrated at specific price levels
- **Curve** - smooth bell-curve distribution

```typescript
// Calculate optimal liquidity distribution
const addInfo = await sdk.Dlmm.calculateAddLiquidityInfo(params)

// Add liquidity
const payload = await sdk.Dlmm.addLiquidityPayload(params)

// Remove liquidity
const payload = await sdk.Dlmm.removeLiquidityPayload(params)

// Close position
const payload = await sdk.Dlmm.closePositionPayload(params)
```

### Swap Operations

```typescript
// Get swap quote
const quote = await sdk.Dlmm.preSwapQuote(params)

// Execute swap
const payload = await sdk.Dlmm.swapPayload(params)
```

### Fee & Reward Operations

```typescript
// Get total fee rate (base + variable)
const feeRate = await sdk.Dlmm.getTotalFeeRate(pool_id)

// Collect fees
const payload = await sdk.Dlmm.collectFeePayload(params)

// Collect rewards
const payload = await sdk.Dlmm.collectRewardPayload(params)
```

### Pool Creation

```typescript
// Create pool only
const payload = await sdk.Dlmm.createPoolPayload(params)

// Create pool + add initial liquidity
const payload = await sdk.Dlmm.createPoolAndAddLiquidityPayload(params)
```

### Utility: BinUtils

```typescript
import { BinUtils } from '@cetusprotocol/dlmm-sdk'

// Price-bin conversions
BinUtils.getPriceFromBinId(binId, binStep)
BinUtils.getBinIdFromPrice(price, binStep)

// Liquidity calculations
BinUtils.calculateLiquidity(params)
```

---

## Vaults SDK

Automated liquidity management with fee reinvestment and rebalancing.

```bash
npm install @cetusprotocol/vaults-sdk
```

### Initialization

```typescript
import { CetusVaultsSDK } from '@cetusprotocol/vaults-sdk'

const sdk = CetusVaultsSDK.createSDK()
// or: CetusVaultsSDK.createSDK({ env, sui_client })
// or: CetusVaultsSDK.createSDK({ env, full_rpc_url })
sdk.setSenderAddress(wallet)
```

### Vault Queries

```typescript
// Get all vaults for an owner
const vaults = await sdk.Vaults.getOwnerVaultsBalance(owner)

// Get specific vault
const vault = await sdk.Vaults.getVault(vault_id)

// Get LP token balance
const balance = await sdk.getOwnerCoinBalances(address, lp_token_type)
```

### Deposit

```typescript
// Calculate deposit amounts
const amounts = await sdk.Vaults.calculateDepositAmount(params)

// Execute deposit
const tx = await sdk.Vaults.deposit(params, tx)
```

### Withdraw

```typescript
// Calculate withdrawal amounts
const amounts = await sdk.Vaults.calculateWithdrawAmount(params)

// Execute withdrawal
const tx = await sdk.Vaults.withdraw(params, tx)
```

### Vesting

```typescript
// Get vest info for multiple vaults
const vestInfoList = await sdk.Vest.getVaultsVestInfoList([vaultId])

// Get vest info for a single vault
const vestInfo = await sdk.Vest.getVaultsVestInfo(vault_id)

// Get user's vest NFTs
const nfts = await sdk.Vest.getOwnerVaultVestNFT(senderAddress)

// Redeem vested tokens
const payload = await sdk.Vest.buildRedeemPayload(options)
```

---

## Farms SDK

Staking CLMM positions for additional reward farming.

```bash
npm install @cetusprotocol/farms-sdk
```

### Initialization

```typescript
import { CetusFarmsSDK } from '@cetusprotocol/farms-sdk'

const sdk = CetusFarmsSDK.createSDK()
// or: CetusFarmsSDK.createSDK({ env, sui_client })
sdk.setSenderAddress(wallet)
```

### Pool Queries

```typescript
// Get all farming pools
const pools = await sdk.Farms.getFarmsPoolList()

// Get specific pool
const pool = await sdk.Farms.getFarmsPool(pool_id)

// Get user's position NFTs
const nfts = await sdk.Farms.getOwnedFarmsPositionNFTList(wallet)

// Get specific NFT details
const nft = await sdk.Farms.getFarmsPositionNFT(position_nft_id)
```

### Staking Operations

```typescript
// Stake a CLMM position into farm
const payload = await sdk.Farms.depositPayload({ pool_id, clmm_position_id })

// Unstake position from farm
const payload = await sdk.Farms.withdrawPayload({ pool_id, position_nft_id })

// Harvest rewards
const payload = await sdk.Farms.harvestPayload({ pool_id, position_nft_id })

// Batch harvest + collect CLMM fees
const payload = await sdk.Farms.batchHarvestAndClmmFeePayload(farms_list, clmm_list)
```

### Liquidity Operations (within Farm)

```typescript
// Add liquidity with fixed coin amount
const payload = await sdk.Farms.addLiquidityFixCoinPayload(params)

// Remove liquidity
const payload = await sdk.Farms.removeLiquidityPayload(params)

// Claim fees and CLMM rewards
const payload = await sdk.Farms.claimFeeAndClmmReward({ pool_id, position_nft_id })
```

### Error Codes

| Code | Description |
|------|-------------|
| 1 | Invalid CLMM Pool ID |
| 2 | Invalid Position NFT |
| ... | ... |
| 15 | Amount Out Below Min Limit |

---

## xCETUS SDK

Platform equity token management - convert CETUS to non-transferable xCETUS for governance and rewards.

```bash
npm install @cetusprotocol/xcetus-sdk
```

### Initialization

```typescript
import { CetusXcetusSDK } from '@cetusprotocol/xcetus-sdk'

const sdk = CetusXcetusSDK.createSDK({ env: 'mainnet', sui_client })
sdk.setSenderAddress(wallet)
```

### Data Retrieval

```typescript
// Get user's veNFT (holds xCETUS balance)
const veNFT = await sdk.Xcetus.getOwnerVeNFT()

// Get user's active locks
const locks = await sdk.Xcetus.getOwnerRedeemLockList()

// Get dividend manager info
const dividendMgr = await sdk.Xcetus.getDividendManager()

// Get veNFT dividend info
const dividendInfo = await sdk.Xcetus.getVeNFTDividendInfo()

// Get xCETUS manager (for ratio calculations)
const manager = await sdk.Xcetus.getXcetusManager()
```

### Token Operations

```typescript
// Convert CETUS -> xCETUS (1:1 ratio)
const payload = await sdk.Xcetus.convertPayload(params)

// Start lock-up redemption (xCETUS -> CETUS, time-locked)
const payload = await sdk.Xcetus.redeemLockPayload(params)

// Complete redemption after lock expires
const payload = await sdk.Xcetus.redeemPayload(params)

// Cancel active lock
const payload = await sdk.Xcetus.cancelRedeemPayload(params)

// Claim accumulated dividends
const payload = await sdk.Xcetus.redeemDividendV3Payload(params)
```

### Utility Functions

```typescript
// Calculate redeemable CETUS for given lock duration
const amount = sdk.Xcetus.redeemNum(lockDays, xCetusAmount)

// Reverse calculation
const xAmount = sdk.Xcetus.reverseRedeemNum(lockDays, cetusAmount)

// Get historical dividend data
const phaseInfo = await sdk.Xcetus.getPhaseDividendInfo(phase)

// Check lock status
import { XCetusUtil } from '@cetusprotocol/xcetus-sdk'
const isLocked = XCetusUtil.isLocked(lockObj)
```

---

## Limit Order SDK

Place limit orders with specified price and expiration.

```bash
npm install @cetusprotocol/limit-sdk
```

### Initialization

```typescript
import { CetusLimitSDK } from '@cetusprotocol/limit-sdk'

const sdk = CetusLimitSDK.createSDK()
// or: CetusLimitSDK.createSDK({ env, sui_client })
sdk.setSenderAddress(wallet)
```

### Order Management

```typescript
// Place a limit order
const payload = await sdk.Limit.placeLimitOrder(params)
// params: coin types, amount, price, expiration

// Cancel running orders
const payload = await sdk.Limit.cancelOrdersByOwner(params)

// Claim completed order proceeds
const payload = await sdk.Limit.claimTargetCoin(params)
```

### Order Queries

```typescript
// Get order details
const order = await sdk.Limit.getLimitOrder(orderId)

// Get all orders for a wallet
const orders = await sdk.Limit.getOwnerLimitOrderList(address)

// Get order operation logs
const logs = await sdk.Limit.getLimitOrderLogs(orderId)

// Get claim logs
const claimLogs = await sdk.Limit.getLimitOrderClaimLogs(orderId)
```

### Pool Info

```typescript
// Get supported tokens
const tokens = await sdk.Limit.getLimitOrderTokenList()

// Get available pools
const pools = await sdk.Limit.getLimitOrderPoolList()

// Get specific pool
const pool = await sdk.Limit.getLimitOrderPool(coinA, coinB)

// Get pool indexer ID
const indexerId = await sdk.Limit.getPoolIndexerId(coinA, coinB)
```

### Execution

```typescript
// Execute transaction
await sdk.FullClient.executeTx(keyPair, payload, true)
```

Order statuses: `Running` | `PartialCompleted` | `Completed` | `Cancelled`

---

## Burn SDK

Permanently lock liquidity positions while still earning fees and rewards.

```bash
npm install @cetusprotocol/burn-sdk
```

### Initialization

```typescript
import { CetusBurnSDK } from '@cetusprotocol/burn-sdk'

const sdk = CetusBurnSDK.createSDK()
// or: CetusBurnSDK.createSDK({ env, sui_client })
sdk.setSenderAddress(wallet)
```

### Queries

```typescript
// Get burn pool list
const pools = await sdk.Burn.getBurnPoolList()

// Get burn positions for a pool
const positions = await sdk.Burn.getPoolBurnPositionList(pool_id)

// Get burn positions for an account
const posIds = await sdk.Burn.getBurnPositionList(account_address)

// Get position details
const pos = await sdk.Burn.getBurnPosition(pos_id)
```

### Burn Operations

```typescript
// Lock liquidity permanently (irreversible!)
const payload = await sdk.Burn.createBurnPayload(params)

// Burn LP v2 (auto-validates, no pool object needed)
const payload = await sdk.Burn.createBurnLPV2Payload(pos_id)
```

### Fee & Reward Collection (still works after burn)

```typescript
// Collect fees for single position
const payload = await sdk.Burn.createCollectFeePayload(params)

// Collect rewards for single position
const payload = await sdk.Burn.createCollectRewardPayload(params)

// Batch collect fees for multiple positions
const payload = await sdk.Burn.createCollectFeesPayload(params)

// Batch collect rewards for multiple positions
const payload = await sdk.Burn.createCollectRewardsPayload(params)
```

### Vesting

```typescript
// Redeem vested tokens
const payload = await sdk.Burn.redeemVestPayload(params)
// params: versioned_id, vester_id, pool_data, period
```

---

## DCA SDK

Dollar-Cost Averaging - automated periodic token purchases.

```bash
npm install @cetusprotocol/dca-sdk
```

### Initialization

```typescript
import { CetusDcaSDK } from '@cetusprotocol/dca-sdk'

const sdk = CetusDcaSDK.createSDK()
// or: CetusDcaSDK.createSDK({ env, sui_client })
sdk.setSenderAddress(wallet)
```

### Order Management

```typescript
// Create DCA order
const payload = await sdk.Dca.dcaOpenOrderPayload({
  // coin types, total amount, per-cycle amount,
  // cycle frequency, min/max price bounds
})

// Get order details
const order = await sdk.Dca.getDcaOrders(orderId)

// Get order transaction history
const deals = await sdk.Dca.getDcaOrdersMakeDeal(orderId)

// Withdraw from DCA order
const payload = await sdk.Dca.withdrawPayload(params)

// Close one or multiple DCA orders
const payload = await sdk.Dca.dcaCloseOrderPayload(params)
```

### Token Whitelist

```typescript
// Get supported tokens
const whitelist = await sdk.Dca.getDcaCoinWhiteList()
```

Whitelist modes:
| Mode | Description |
|------|-------------|
| 0 | Disabled |
| 1 | in_coin only |
| 2 | out_coin only |
| 3 | Both coin types enabled |

### Execution

```typescript
await sdk.FullClient.sendTransaction(keyPair, payload)
```

---

## Zap SDK

One-click liquidity operations - add/remove liquidity with flexible input modes.

```bash
npm install @cetusprotocol/zap-sdk
```

### Initialization

```typescript
import { CetusZapSDK } from '@cetusprotocol/zap-sdk'

const sdk = CetusZapSDK.createSDK()
// or: CetusZapSDK.createSDK({ env, sui_client, full_rpc_url })
sdk.setSenderAddress(wallet)
```

### Deposit (Add Liquidity)

Deposit modes: `FixedOneSide` | `FlexibleBoth` | `OnlyCoinA` | `OnlyCoinB`

```typescript
// Pre-calculate deposit amounts
const calcResult = await sdk.Zap.preCalculateDepositAmount({
  pool_id,
  tick_lower,
  tick_upper,
  current_sqrt_price,
  slippage,
  coin_type_a,
  coin_type_b,
  decimals_a,
  decimals_b,
  mode,
  amount_a, // or amount_b depending on mode
})

// Build and execute deposit
const payload = await sdk.Zap.buildDepositPayload({
  ...calcResult,
  // optional: existing position_id to add to
})
```

### Withdraw (Remove Liquidity)

```typescript
// Pre-calculate withdrawal amounts
const calcResult = await sdk.Zap.preCalculateWithdrawAmount(params)

// Build withdrawal transaction
const payload = await sdk.Zap.buildWithdrawPayload({
  ...calcResult,
  collect_fee: true,    // optionally collect fees
  collect_reward: true, // optionally collect rewards
})
```

---

## Aggregator SDK

Multi-DEX swap aggregator optimizing trades across Cetus, DeepBook, Kriya, FlowX, Aftermath, and more.

```bash
npm install @cetusprotocol/aggregator-sdk
```

### Workflow

```typescript
import { CetusAggregatorSDK } from '@cetusprotocol/aggregator-sdk'

// Step 1: Initialize client
const client = CetusAggregatorSDK.createSDK({
  env: 'mainnet',
  // RPC and package config
})

// Step 2: Find optimal route
const routes = await client.findRouters({
  coinTypeFrom,
  coinTypeTo,
  amount,
})

// Step 3a: Fast swap (simple)
const result = await client.fastRouterSwap({
  routes,
  slippage, // e.g. 0.01 for 1%
  keyPair,
})

// Step 3b: Build PTB transaction (advanced)
const tx = await client.routerSwap({
  routes,
  slippage,
  // manage coin transfers manually
})
```

### Supported DEXs

Cetus, DeepBook, Kriya, FlowX, Aftermath, Turbos, Bluefin, and more.

### Mainnet Contract Addresses

- **CetusAggregatorV2** - Primary aggregator
- **CetusAggregatorV2ExtendV1** - Extended functionality
- **CetusAggregatorV2ExtendV2** - Extended functionality v2

---

## Common Patterns

### SDK Initialization (all packages follow this pattern)

```typescript
// Default mainnet
const sdk = Cetus<Module>SDK.createSDK()

// Custom env
const sdk = Cetus<Module>SDK.createSDK({ env: 'testnet' })

// Custom RPC
const sdk = Cetus<Module>SDK.createSDK({ env: 'mainnet', full_rpc_url: 'YOUR_URL' })

// Custom SuiClient
const sdk = Cetus<Module>SDK.createSDK({ env: 'mainnet', sui_client: yourClient })

// Always set sender before transactions
sdk.setSenderAddress('0x...')

// Update RPC at runtime
sdk.updateFullRpcUrl('NEW_URL')
```

### Transaction Execution

```typescript
// Using FullClient
await sdk.FullClient.executeTx(keyPair, payload, true)

// Or sendTransaction
await sdk.FullClient.sendTransaction(keyPair, payload)
```

---

## Package Reference

| Package | npm | Purpose |
|---------|-----|---------|
| common | `@cetusprotocol/common-sdk` | Shared utilities |
| clmm | `@cetusprotocol/sui-clmm-sdk` | Concentrated liquidity AMM |
| dlmm | `@cetusprotocol/dlmm-sdk` | Dynamic liquidity (bin-based) |
| vaults | `@cetusprotocol/vaults-sdk` | Automated vault management |
| farms | `@cetusprotocol/farms-sdk` | Yield farming |
| xcetus | `@cetusprotocol/xcetus-sdk` | Governance token (xCETUS) |
| limit | `@cetusprotocol/limit-sdk` | Limit orders |
| burn | `@cetusprotocol/burn-sdk` | Permanent liquidity lock |
| dca | `@cetusprotocol/dca-sdk` | Dollar-cost averaging |
| zap | `@cetusprotocol/zap-sdk` | One-click liquidity |
| aggregator | `@cetusprotocol/aggregator-sdk` | Multi-DEX swap routing |
