#!/usr/bin/env python3
"""
Canvas HTML Injection Helper

Workaround for Canvas file path restrictions.
Use this when file:// URLs and data URLs don't work.
"""

def inject_html_to_canvas(html_content: str, node_name: str = "Dushyant's Mac mini"):
    """
    Inject HTML directly into Canvas via document.write()
    
    Args:
        html_content: The complete HTML document as a string
        node_name: Name of the node (default: "Dushyant's Mac mini")
    
    Returns:
        The canvas tool commands as a dictionary
    """
    # Escape backticks in HTML (they break the JS template literal)
    html_escaped = html_content.replace('`', '\\`')
    
    # JavaScript to inject HTML
    js_code = f"""document.open();
document.write(`{html_escaped}`);
document.close();"""
    
    return {
        "step1_present": {
            "action": "present",
            "url": "about:blank",
            "target": node_name
        },
        "step2_inject": {
            "action": "eval",
            "javaScript": js_code,
            "target": node_name
        }
    }


def example_usage():
    """
    Example: Show a simple dashboard on Canvas
    """
    html = """<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Example Dashboard</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 40px;
            min-height: 100vh;
        }
        .card {
            background: white;
            border-radius: 16px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            max-width: 600px;
            margin: 0 auto;
        }
        h1 { color: #667eea; margin-bottom: 20px; }
        p { color: #666; line-height: 1.6; }
    </style>
</head>
<body>
    <div class="card">
        <h1>🎨 Canvas Injection Works!</h1>
        <p>This HTML was injected directly via document.write()</p>
        <p>No file paths, no localhost server needed.</p>
    </div>
</body>
</html>"""
    
    commands = inject_html_to_canvas(html)
    
    print("Step 1: Present blank canvas")
    print(commands["step1_present"])
    print("\nStep 2: Inject HTML")
    print(commands["step2_inject"])


if __name__ == "__main__":
    example_usage()
