# Canvas Loading Reference

## The Problem

Canvas on Mac mini **blocks file path access** for security (sandboxed browser context).

**❌ These DON'T work:**
```python
# File paths are blocked
canvas.present(url="file:///tmp/dashboard.html")

# Data URLs are unreliable
canvas.present(url="data:text/html;base64,...")
```

**✅ These DO work:**
```python
# Method 1: Localhost server
canvas.present(url="http://localhost:9876/")

# Method 2: Direct HTML injection (the workaround)
canvas.present(url="about:blank")
canvas.eval(javaScript='document.open(); document.write(`<html>...</html>`); document.close();')
```

## Method 2: Direct Injection (Step-by-Step)

### 1. Load your HTML
```python
html_content = """<!DOCTYPE html>
<html>
<head>
    <style>
        body { background: #667eea; color: white; padding: 40px; }
    </style>
</head>
<body>
    <h1>Hello Canvas!</h1>
</body>
</html>"""
```

### 2. Build the injection JavaScript
```python
# IMPORTANT: Escape backticks in your HTML!
html_escaped = html_content.replace('`', '\\`')

js_code = f"""document.open();
document.write(`{html_escaped}`);
document.close();"""
```

### 3. Execute via canvas tool
```python
from tools import canvas

# Step 1: Present blank page
canvas(action="present", url="about:blank", target="Dushyant's Mac mini")

# Step 2: Inject HTML
canvas(action="eval", javaScript=js_code, target="Dushyant's Mac mini")
```

## When to Use Each Method

**Use localhost (Method 1) when:**
- App has multiple files (JS, CSS, images)
- App needs to load external resources
- App is complex and reused often

**Use direct injection (Method 2) when:**
- Quick one-off displays
- Self-contained HTML (inline CSS/JS)
- Demoing skills (like Substack CLI)
- File path access is blocked

## Common Mistakes

### ❌ Forgetting to escape backticks
```python
# WRONG - will break if HTML contains backticks
js = f'document.write(`{html}`);'

# RIGHT - escape them first
js = f'document.write(`{html.replace("`", "\\`")}`);'
```

### ❌ Using file paths
```python
# WRONG - blocked by Canvas security
canvas.navigate(url="file:///tmp/app.html")

# RIGHT - serve via localhost
canvas.navigate(url="http://localhost:9876/")
```

### ❌ Not using about:blank first
```python
# WRONG - may fail if Canvas is on another page
canvas.eval(javaScript="document.write(...)")

# RIGHT - reset to blank first
canvas.present(url="about:blank")
canvas.eval(javaScript="document.write(...)")
```

## Helper Script

Use `canvas-inject.py` to automate the pattern:

```python
from canvas_inject import inject_html_to_canvas

html = open("my-dashboard.html").read()
commands = inject_html_to_canvas(html, node_name="Dushyant's Mac mini")

# Returns dict with step1_present and step2_inject commands
```

## Why This Matters

Canvas is **sandboxed** for security — it can't read arbitrary files from your filesystem. This is by design to prevent malicious content from accessing local data.

The workaround (document.write via eval) works because:
1. You're not asking Canvas to read a file
2. You're sending the HTML as a string through OpenClaw's trusted channel
3. Canvas renders it in the sandboxed browser context

Think of it like copy-paste instead of file reading.
