#!/usr/bin/env python3
"""
Camoufox stealth browser fetch.
Maximum anti-bot evasion - C++ level Firefox patches.
Best for: Yelp, Datadome, aggressive Cloudflare Turnstile.

Usage:
    python camoufox-fetch.py "https://example.com" [options]

Options:
    --wait N          Wait N seconds after page load (default: 8)
    --screenshot FILE Save screenshot to file
    --output FILE     Save HTML to file
    --proxy URL       Use proxy (http://user:pass@host:port)
    --headless        Run headless (still stealthy with Camoufox)
"""

import asyncio
import argparse
import sys

try:
    from camoufox.async_api import AsyncCamoufox
except ImportError:
    print("Error: camoufox not installed. Run:")
    print("  pip install camoufox")
    print("  python -c 'import camoufox; camoufox.install()'")
    sys.exit(1)


async def fetch_page(url: str, wait: int = 8, screenshot: str = None,
                     output: str = None, proxy: str = None, headless: bool = False):
    """Fetch a page using Camoufox stealth browser."""
    
    print(f"🥷 Starting Camoufox browser (max stealth)...")
    
    # Camoufox config
    config = {
        "headless": headless,
    }
    
    if proxy:
        # Parse proxy URL
        if "@" in proxy:
            # Has auth: http://user:pass@host:port
            auth_part = proxy.split("@")[0].replace("http://", "").replace("https://", "")
            host_part = proxy.split("@")[1]
            user, password = auth_part.split(":")
            host, port = host_part.split(":")
            config["proxy"] = {
                "server": f"http://{host}:{port}",
                "username": user,
                "password": password,
            }
        else:
            config["proxy"] = {"server": proxy}
    
    async with AsyncCamoufox(**config) as browser:
        page = await browser.new_page()
        
        print(f"📡 Navigating to: {url}")
        await page.goto(url, wait_until="domcontentloaded")
        
        # Wait for anti-bot to resolve
        print(f"⏳ Waiting {wait}s for anti-bot resolution...")
        await asyncio.sleep(wait)
        
        # Get page info
        title = await page.title()
        print(f"📄 Page title: {title}")
        
        # Get content
        content = await page.content()
        
        # Check for block indicators
        if "Access Denied" in content or "blocked" in content.lower():
            print("⚠️  Warning: Page may still be blocked. Check proxy quality.")
        elif "challenge" in content.lower() and "cloudflare" in content.lower():
            print("⚠️  Warning: Cloudflare challenge detected. Increase wait time.")
        else:
            print("✅ No obvious block indicators detected")
        
        # Screenshot
        if screenshot:
            await page.screenshot(path=screenshot, full_page=True)
            print(f"📸 Screenshot saved: {screenshot}")
        
        # Save HTML
        if output:
            with open(output, 'w', encoding='utf-8') as f:
                f.write(content)
            print(f"💾 HTML saved: {output}")
        
        # Summary
        if not output:
            print(f"\n✅ Success! Page loaded ({len(content)} bytes)")
            print(f"   Final URL: {page.url}")
        
        return content


def main():
    parser = argparse.ArgumentParser(description="Camoufox stealth browser fetch (max evasion)")
    parser.add_argument("url", help="URL to fetch")
    parser.add_argument("--wait", type=int, default=8, help="Wait time in seconds (default: 8)")
    parser.add_argument("--screenshot", help="Save screenshot to file")
    parser.add_argument("--output", help="Save HTML to file")
    parser.add_argument("--proxy", help="Proxy URL (http://user:pass@host:port)")
    parser.add_argument("--headless", action="store_true", help="Run headless")
    
    args = parser.parse_args()
    
    asyncio.run(
        fetch_page(
            url=args.url,
            wait=args.wait,
            screenshot=args.screenshot,
            output=args.output,
            proxy=args.proxy,
            headless=args.headless
        )
    )


if __name__ == "__main__":
    main()
