#!/usr/bin/env python3
"""
TLS-spoofed HTTP client for API scraping.
Mimics real browser TLS fingerprints without running a browser.

Usage:
    python curl-api.py "https://api.example.com/endpoint" [options]

Options:
    --headers JSON    JSON string of headers
    --method METHOD   HTTP method (GET, POST, etc.)
    --data DATA       Request body (for POST/PUT)
    --impersonate VER Browser to impersonate (chrome120, safari17, etc.)
    --proxy URL       Proxy URL
    --output FILE     Save response to file
"""

import argparse
import json
import sys

try:
    from curl_cffi import requests
except ImportError:
    print("Error: curl_cffi not installed. Run: pip install curl_cffi")
    sys.exit(1)


# Available browser impersonations
BROWSERS = [
    "chrome99", "chrome100", "chrome101", "chrome104", "chrome107", 
    "chrome110", "chrome116", "chrome119", "chrome120", "chrome123",
    "chrome99_android", "chrome131",
    "edge99", "edge101",
    "safari15_3", "safari15_5", "safari17_0", "safari17_2_ios",
    "firefox109", "firefox117", "firefox120", "firefox133",
]


def fetch_api(url: str, headers: dict = None, method: str = "GET",
              data: str = None, impersonate: str = "chrome120",
              proxy: str = None, output: str = None):
    """Fetch API endpoint with TLS fingerprint spoofing."""
    
    print(f"🥷 curl_cffi request (impersonating {impersonate})")
    print(f"📡 {method} {url}")
    
    # Validate impersonate
    if impersonate not in BROWSERS:
        print(f"⚠️  Unknown browser '{impersonate}'. Using chrome120.")
        print(f"   Available: {', '.join(BROWSERS[:5])}...")
        impersonate = "chrome120"
    
    # Build request kwargs
    kwargs = {
        "impersonate": impersonate,
    }
    
    if headers:
        kwargs["headers"] = headers
    
    if proxy:
        kwargs["proxies"] = {"http": proxy, "https": proxy}
    
    if data:
        kwargs["data"] = data
    
    # Make request
    try:
        if method.upper() == "GET":
            response = requests.get(url, **kwargs)
        elif method.upper() == "POST":
            response = requests.post(url, **kwargs)
        elif method.upper() == "PUT":
            response = requests.put(url, **kwargs)
        elif method.upper() == "DELETE":
            response = requests.delete(url, **kwargs)
        else:
            response = requests.request(method.upper(), url, **kwargs)
    except Exception as e:
        print(f"❌ Request failed: {e}")
        sys.exit(1)
    
    print(f"📊 Status: {response.status_code}")
    print(f"📏 Response size: {len(response.content)} bytes")
    
    # Check for blocks
    if response.status_code == 403:
        print("⚠️  403 Forbidden - Try different impersonate or residential proxy")
    elif response.status_code == 503:
        print("⚠️  503 Service Unavailable - Likely anti-bot block")
    elif "Access Denied" in response.text:
        print("⚠️  Access Denied in response body")
    
    # Content type
    content_type = response.headers.get("content-type", "")
    print(f"📄 Content-Type: {content_type}")
    
    # Output
    if output:
        mode = "wb" if "octet-stream" in content_type else "w"
        content = response.content if mode == "wb" else response.text
        with open(output, mode) as f:
            f.write(content)
        print(f"💾 Saved to: {output}")
    else:
        # Pretty print JSON if applicable
        if "json" in content_type:
            try:
                data = response.json()
                print("\n📋 Response (JSON):")
                print(json.dumps(data, indent=2)[:2000])
                if len(response.text) > 2000:
                    print(f"... ({len(response.text) - 2000} more bytes)")
            except:
                print("\n📋 Response (raw):")
                print(response.text[:2000])
        else:
            print("\n📋 Response (first 500 chars):")
            print(response.text[:500])
    
    return response


def main():
    parser = argparse.ArgumentParser(description="TLS-spoofed API fetch")
    parser.add_argument("url", help="URL to fetch")
    parser.add_argument("--headers", help="JSON string of headers")
    parser.add_argument("--method", default="GET", help="HTTP method (default: GET)")
    parser.add_argument("--data", help="Request body for POST/PUT")
    parser.add_argument("--impersonate", default="chrome120", 
                        help=f"Browser to impersonate (default: chrome120)")
    parser.add_argument("--proxy", help="Proxy URL")
    parser.add_argument("--output", help="Save response to file")
    
    args = parser.parse_args()
    
    # Parse headers JSON
    headers = None
    if args.headers:
        try:
            headers = json.loads(args.headers)
        except json.JSONDecodeError:
            print(f"❌ Invalid headers JSON: {args.headers}")
            sys.exit(1)
    
    fetch_api(
        url=args.url,
        headers=headers,
        method=args.method,
        data=args.data,
        impersonate=args.impersonate,
        proxy=args.proxy,
        output=args.output
    )


if __name__ == "__main__":
    main()
