#!/usr/bin/env python3
"""
Calorie Counter - Simple, reliable calorie tracking with SQLite
"""

import sqlite3
import sys
from datetime import date
from pathlib import Path

# Database path relative to skill directory
SKILL_DIR = Path(__file__).parent.parent
DB_PATH = SKILL_DIR / "calorie_data.db"


def init_db():
    """Initialize database with schema"""
    conn = sqlite3.connect(DB_PATH)
    c = conn.cursor()

    # Food entries table - simplified
    c.execute('''
        CREATE TABLE IF NOT EXISTS entries (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            date TEXT NOT NULL,
            food_name TEXT NOT NULL,
            calories INTEGER NOT NULL,
            protein INTEGER NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ''')

    # Daily goal - single value
    c.execute('''
        CREATE TABLE IF NOT EXISTS daily_goal (
            id INTEGER PRIMARY KEY CHECK (id = 1),
            calorie_goal INTEGER NOT NULL
        )
    ''')

    # Weight log - pounds, no notes
    c.execute('''
        CREATE TABLE IF NOT EXISTS weight_log (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            date TEXT NOT NULL,
            weight_lbs REAL NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ''')

    # Create indexes
    c.execute('CREATE INDEX IF NOT EXISTS idx_entries_date ON entries(date)')
    c.execute('CREATE INDEX IF NOT EXISTS idx_weight_date ON weight_log(date)')

    conn.commit()
    conn.close()


def get_db():
    """Get database connection"""
    if not DB_PATH.exists():
        init_db()
    return sqlite3.connect(DB_PATH)


def add_entry(food_name, calories, protein):
    """Add a food entry"""
    try:
        calories = int(calories)
        protein = int(protein)
        if calories < 0 or protein < 0:
            print("Error: Calories and protein cannot be negative")
            return False
    except ValueError:
        print("Error: Calories and protein must be numbers")
        return False

    conn = get_db()
    c = conn.cursor()

    today = date.today().isoformat()

    c.execute('''
        INSERT INTO entries (date, food_name, calories, protein)
        VALUES (?, ?, ?, ?)
    ''', (today, food_name, calories, protein))

    entry_id = c.lastrowid
    conn.commit()

    # Totals + goal (so callers don't need to do math)
    c.execute('''
        SELECT COALESCE(SUM(calories), 0), COALESCE(SUM(protein), 0), COUNT(*)
        FROM entries
        WHERE date = ?
    ''', (today,))
    total_calories, total_protein, entry_count = c.fetchone()

    c.execute('SELECT calorie_goal FROM daily_goal WHERE id = 1')
    goal_row = c.fetchone()
    goal = goal_row[0] if goal_row else None

    conn.close()

    print(f"✓ Added: {food_name} ({calories} cal, {protein}g protein)")
    print(f"  Entry ID: {entry_id}")

    if goal is not None:
        remaining = goal - total_calories
        print(
            f"  Today: {total_calories} / {goal} cal (remaining: {remaining})"
            f" | Protein today: {total_protein}g | Entries: {entry_count}"
        )
    else:
        print(
            f"  Today: {total_calories} cal"
            f" | Protein today: {total_protein}g | Entries: {entry_count} | Goal: not set"
        )

    return True


def delete_entry(entry_id):
    """Delete an entry"""
    try:
        entry_id = int(entry_id)
    except ValueError:
        print("Error: Entry ID must be a number")
        return False

    conn = get_db()
    c = conn.cursor()

    # Check if entry exists
    c.execute('SELECT food_name, calories, protein FROM entries WHERE id = ?', (entry_id,))
    row = c.fetchone()

    if not row:
        print(f"Error: Entry ID {entry_id} not found")
        conn.close()
        return False

    food_name, calories, protein = row

    c.execute('DELETE FROM entries WHERE id = ?', (entry_id,))
    conn.commit()
    conn.close()

    print(f"✓ Deleted entry {entry_id}: {food_name} ({calories} cal, {protein}g protein)")
    return True


def set_goal(calories):
    """Set calorie goal"""
    try:
        calories = int(calories)
        if calories <= 0:
            print("Error: Goal must be positive")
            return False
    except ValueError:
        print("Error: Calories must be a number")
        return False

    conn = get_db()
    c = conn.cursor()

    c.execute('''
        INSERT OR REPLACE INTO daily_goal (id, calorie_goal)
        VALUES (1, ?)
    ''', (calories,))

    conn.commit()
    conn.close()

    print(f"✓ Set daily goal: {calories} cal")
    return True


def get_goal():
    """Get calorie goal"""
    conn = get_db()
    c = conn.cursor()

    c.execute('SELECT calorie_goal FROM daily_goal WHERE id = 1')
    row = c.fetchone()
    conn.close()

    return row[0] if row else None


def list_entries(target_date=None):
    """List entries for a date"""
    if target_date is None:
        target_date = date.today().isoformat()

    conn = get_db()
    c = conn.cursor()

    c.execute('''
        SELECT id, food_name, calories, protein,
               strftime('%H:%M', created_at) as time
        FROM entries
        WHERE date = ?
        ORDER BY created_at
    ''', (target_date,))

    rows = c.fetchall()
    conn.close()

    if not rows:
        print(f"No entries for {target_date}")
        return []

    print(f"\nEntries for {target_date}:")
    print("-" * 60)

    for entry_id, food_name, calories, protein, time in rows:
        print(f"ID {entry_id:3d} | {time} | {food_name:20s} | {calories:4d} cal | {protein:3d}g protein")

    print("-" * 60)
    return rows


def summary(target_date=None):
    """Show daily summary"""
    if target_date is None:
        target_date = date.today().isoformat()

    conn = get_db()
    c = conn.cursor()

    # Get entries
    c.execute('''
        SELECT SUM(calories), SUM(protein), COUNT(*)
        FROM entries
        WHERE date = ?
    ''', (target_date,))

    result = c.fetchone()
    total_calories = result[0] or 0
    total_protein = result[1] or 0
    entry_count = result[2] or 0

    # Get goal
    goal = get_goal()

    conn.close()

    print(f"\n{'='*60}")
    print(f"DAILY SUMMARY - {target_date}")
    print(f"{'='*60}")
    print(f"Entries: {entry_count}")
    print(f"Total consumed: {total_calories} cal | {total_protein}g protein")

    if goal:
        remaining = goal - total_calories
        print(f"Daily goal: {goal} cal")
        print(f"Remaining: {remaining} cal")

        if remaining < 0:
            print(f"  ⚠️  Over goal by {abs(remaining)} cal")
        elif remaining == 0:
            print(f"  ✓ Goal reached exactly!")
        else:
            percent = (total_calories / goal) * 100
            print(f"  {percent:.1f}% of goal consumed")
    else:
        print("Daily goal: Not set (use 'goal' command)")

    print(f"{'='*60}\n")

    # Show entries if any
    if entry_count > 0:
        list_entries(target_date)


def log_weight(weight_lbs):
    """Log weight"""
    try:
        weight_lbs = float(weight_lbs)
        if weight_lbs <= 0:
            print("Error: Weight must be positive")
            return False
    except ValueError:
        print("Error: Weight must be a number")
        return False

    conn = get_db()
    c = conn.cursor()

    today = date.today().isoformat()

    c.execute('''
        INSERT INTO weight_log (date, weight_lbs)
        VALUES (?, ?)
    ''', (today, weight_lbs))

    conn.commit()
    conn.close()

    print(f"✓ Logged weight: {weight_lbs} lbs")
    return True


def weight_history(days=30):
    """Show weight history"""
    conn = get_db()
    c = conn.cursor()

    c.execute('''
        SELECT date, weight_lbs
        FROM weight_log
        ORDER BY date DESC
        LIMIT ?
    ''', (days,))

    rows = c.fetchall()
    conn.close()

    if not rows:
        print("No weight log entries")
        return

    print(f"\nWeight History (last {days} days):")
    print("-" * 40)

    for date_str, weight_lbs in rows:
        print(f"{date_str} | {weight_lbs:.1f} lbs")

    # Calculate change
    if len(rows) >= 2:
        first_weight = rows[-1][1]
        last_weight = rows[0][1]
        change = last_weight - first_weight

        print("-" * 40)
        if change > 0:
            print(f"Change: +{change:.1f} lbs")
        elif change < 0:
            print(f"Change: {change:.1f} lbs")
        else:
            print(f"Change: No change")

    print()


def history(days=7):
    """Show calorie history"""
    conn = get_db()
    c = conn.cursor()

    c.execute('''
        SELECT date, SUM(calories) as total, SUM(protein) as total_protein
        FROM entries
        WHERE date >= date('now', '-' || ? || ' days')
        GROUP BY date
        ORDER BY date DESC
    ''', (days,))

    rows = c.fetchall()
    conn.close()

    if not rows:
        print(f"No entries in last {days} days")
        return

    print(f"\nCalorie History (last {days} days):")
    print("-" * 60)

    goal = get_goal()

    for date_str, total, total_protein in rows:
        if goal:
            remaining = goal - total
            status = f"({remaining:+d} remaining)" if remaining != 0 else "(goal met)"
        else:
            status = "(no goal)"

        goal_str = str(goal) if goal else "N/A"
        print(f"{date_str} | {total:4d} cal | {total_protein:3d}g protein | Goal: {goal_str:>4s} {status}")

    print()


def usage():
    """Print usage information"""
    print("""
Calorie Counter - Usage

Commands:
  add <food> <calories> <protein>   Add food entry
  delete <id>                       Delete entry
  list                              List today's entries
  summary                           Show today's summary
  goal <calories>                   Set daily goal
  weight <lbs>                      Log weight in pounds
  weight-history [days]             Show weight history (default 30)
  history [days]                    Show calorie history (default 7)

Examples:
  add "chicken breast" 165 31
  delete 42
  goal 2000
  weight 165.5
""")


def main():
    if len(sys.argv) < 2:
        usage()
        sys.exit(1)

    command = sys.argv[1]

    try:
        if command == "add":
            if len(sys.argv) < 5:
                print("Error: add requires <food> <calories> <protein>")
                sys.exit(1)
            food = sys.argv[2]
            calories = sys.argv[3]
            protein = sys.argv[4]
            add_entry(food, calories, protein)

        elif command == "delete":
            if len(sys.argv) < 3:
                print("Error: delete requires <id>")
                sys.exit(1)
            delete_entry(sys.argv[2])

        elif command == "list":
            list_entries()

        elif command == "summary":
            summary()

        elif command == "goal":
            if len(sys.argv) < 3:
                print("Error: goal requires <calories>")
                sys.exit(1)
            set_goal(sys.argv[2])

        elif command == "weight":
            if len(sys.argv) < 3:
                print("Error: weight requires <lbs>")
                sys.exit(1)
            log_weight(sys.argv[2])

        elif command == "weight-history":
            days = int(sys.argv[2]) if len(sys.argv) > 2 else 30
            weight_history(days)

        elif command == "history":
            days = int(sys.argv[2]) if len(sys.argv) > 2 else 7
            history(days)

        else:
            print(f"Error: Unknown command '{command}'")
            usage()
            sys.exit(1)

    except Exception as e:
        print(f"Error: {e}")
        sys.exit(1)


if __name__ == "__main__":
    main()
