#!/bin/bash
# Build with Public - Helper Script
# 简化博客项目管理和发布流程

set -e

PROJECT_NAME="${1:-codewithriver}"
ACTION="${2:-init}"

show_help() {
    echo "Build with Public - 技术博客写作助手"
    echo ""
    echo "用法:"
    echo "  ./bwp.sh <project-name> init       # 初始化新项目"
    echo "  ./bwp.sh <project-name> start      # 启动服务器"
    echo "  ./bwp.sh <project-name> stop       # 停止服务器"
    echo "  ./bwp.sh <project-name> status     # 查看状态"
    echo "  ./bwp.sh <project-name> commit     # 提交到 Git"
    echo ""
    echo "示例:"
    echo "  ./bwp.sh myblog init"
    echo "  ./bwp.sh myblog start"
}

init_project() {
    echo "🚀 初始化 Build with Public 项目: $PROJECT_NAME"
    
    # 创建目录结构
    mkdir -p ~/$PROJECT_NAME/{articles,draft,images,logs,tweets}
    cd ~/$PROJECT_NAME
    
    # 初始化 Git
    git init
    git config user.email "$(whoami)@example.com"
    git config user.name "$(whoami)"
    
    # 创建 .env
    cat > .env << EOF
# Build with Public - Configuration
PORT=12000
HOST=0.0.0.0
AUTH_USERNAME=user
AUTH_PASSWORD=openskill
CUSTOM_DOMAIN=localhost
EOF
    
    # 复制 server.py 模板
    if [ -f "$(dirname $0)/templates/server.py.template" ]; then
        cp "$(dirname $0)/templates/server.py.template" server.py
        chmod +x server.py
    else
        echo "⚠️  未找到 server.py 模板，请手动创建"
    fi
    
    # 创建 README
    cat > README.md << EOF
# $PROJECT_NAME - Build with Public

技术博客写作项目

## 目录结构
- articles/ - 发布的文章
- draft/ - 写作大纲和草稿
- images/ - 配图
- logs/ - 写作日志

## 启动服务器
python server.py

## 访问地址
http://localhost:12000
EOF
    
    # 首次提交
    git add .
    git commit -m "[$(date +%Y-%m-%d)] init: 初始化 Build with Public 项目"
    
    echo "✅ 项目初始化完成！"
    echo ""
    echo "下一步:"
    echo "  cd ~/$PROJECT_NAME"
    echo "  ./bwp.sh $PROJECT_NAME start"
}

start_server() {
    cd ~/$PROJECT_NAME
    
    if pgrep -f "server.py" > /dev/null; then
        echo "⚠️  服务器已在运行"
        return
    fi
    
    echo "🚀 启动服务器..."
    nohup python server.py > logs/server.log 2>&1 &
    sleep 2
    
    echo "✅ 服务器已启动"
    echo "🔗 访问地址: http://localhost:12000"
    echo "🔐 默认密码: openskill"
}

stop_server() {
    if pgrep -f "server.py" > /dev/null; then
        pkill -f "server.py"
        echo "✅ 服务器已停止"
    else
        echo "⚠️  服务器未运行"
    fi
}

show_status() {
    cd ~/$PROJECT_NAME
    
    echo "📊 项目状态: $PROJECT_NAME"
    echo ""
    
    # Git 状态
    echo "📝 Git 状态:"
    git log --oneline -3 2>/dev/null || echo "  暂无提交"
    echo ""
    
    # 文件统计
    echo "📁 文件统计:"
    echo "  Articles: $(ls articles/*.md 2>/dev/null | wc -l) 篇"
    echo "  Drafts: $(ls draft/*.md 2>/dev/null | wc -l) 个"
    echo ""
    
    # 服务器状态
    if pgrep -f "server.py" > /dev/null; then
        echo "🟢 服务器: 运行中"
        echo "   URL: http://localhost:12000"
    else
        echo "🔴 服务器: 未运行"
    fi
}

git_commit() {
    cd ~/$PROJECT_NAME
    
    echo "📝 提交更改到 Git..."
    git add -A
    
    # 如果有更改才提交
    if ! git diff --cached --quiet; then
        read -p "提交信息: " msg
        git commit -m "[$(date +%Y-%m-%d)] $msg"
        echo "✅ 提交成功"
    else
        echo "ℹ️  没有需要提交的更改"
    fi
}

case "$ACTION" in
    init)
        init_project
        ;;
    start)
        start_server
        ;;
    stop)
        stop_server
        ;;
    status)
        show_status
        ;;
    commit)
        git_commit
        ;;
    help|--help|-h)
        show_help
        ;;
    *)
        echo "未知操作: $ACTION"
        show_help
        exit 1
        ;;
esac
