---
name: build-with-public-writer
description: 系统化构建"Build with Public"技术博客写作工作流，包括目录架构、版本管理、多角度方案、链接分享等完整工具链
---

# Build with Public - 技术博客写作体系

## Overview

本技能帮助你建立一套完整的**公开构建 (Build with Public)** 技术博客写作工作流。从目录架构搭建、Git 版本管理、多角度写作方案设计，到最终的文章链接分享，提供端到端的工具链支持。

**核心价值**：
- 🏗️ 标准化的博客项目架构
- 📝 系统化的多角度写作流程  
- 🔄 版本化的迭代管理机制
- 🌐 即时的链接分享能力

## When to Use

当你需要：
- 建立个人技术博客写作工作流
- 系统化管理多版本技术文章
- 通过公开构建建立技术影响力
- 将写作过程固化为可复用资产

## Prerequisites

- Python 3.11+
- Git
- 一个可访问的域名或 IP（用于分享链接）

---

## 工作流程

### Phase 1: 项目初始化

#### Step 1: 确定项目名称

与用户确认博客项目名称（例如：codewithriver, techblog, buildinpublic 等）：

```
建议名称：codewithriver
建议名称：tech-notes
建议名称：buildlog
```

#### Step 2: 创建目录架构

```bash
mkdir -p ~/{project-name}/{articles,draft,images,logs,tweets}
cd ~/{project-name}
```

**标准目录结构**：
```
{project-name}/
├── articles/          # 发布的文章（按版本管理）
│   ├── 2026-03-02-topic-v1.md
│   ├── 2026-03-02-topic-v2.md
│   └── ...
├── draft/             # 写作大纲和草稿
│   ├── 2026-03-02-topic-outline-01.md
│   └── ...
├── images/            # 文章配图
├── logs/              # 写作日志
├── tweets/            # 推广文案
├── .env               # 配置（端口、密码、域名）
├── server.py          # Web 服务器
└── .git/              # 版本控制
```

#### Step 3: 初始化 Git 仓库

```bash
cd ~/{project-name}
git init
git config user.email "your@email.com"
git config user.name "your-name"
```

#### Step 4: 创建 Web 服务器

创建 `server.py`：

```python
#!/usr/bin/env python3
"""Build with Public - Web Server"""

import os
import mimetypes
from functools import wraps
from flask import Flask, send_file, render_template_string, abort, request, Response
from pathlib import Path
from dotenv import load_dotenv

load_dotenv()

app = Flask(__name__)

CONTENT_DIR = Path(os.environ.get('CONTENT_DIR', os.getcwd()))
PORT = int(os.environ.get('PORT', 12000))
HOST = os.environ.get('HOST', '0.0.0.0')
AUTH_PASSWORD = os.environ.get('AUTH_PASSWORD', 'openskill')
AUTH_USERNAME = os.environ.get('AUTH_USERNAME', 'user')

# [服务器代码模板，此处省略完整实现]
# 完整代码参考 skill 包中的 templates/server.py.template

def check_auth(username, password):
    return username == AUTH_USERNAME and password == AUTH_PASSWORD

def requires_auth(f):
    @wraps(f)
    def decorated(*args, **kwargs):
        auth = request.authorization
        if not auth or not check_auth(auth.username, auth.password):
            return Response('需要认证', 401, {'WWW-Authenticate': 'Basic realm="BuildWithPublic"'})
        return f(*args, **kwargs)
    return decorated

@app.route('/')
@requires_auth
def index():
    # [目录列表实现]
    pass

@app.route('/<path:subpath>')
@requires_auth
def serve_path(subpath):
    # [文件服务实现]
    pass

if __name__ == '__main__':
    print(f"🚀 Server starting on http://{HOST}:{PORT}")
    app.run(host=HOST, port=PORT, debug=False)
```

#### Step 5: 创建 .env 配置文件

```bash
cat > .env << 'EOF'
# Build with Public - Configuration
PORT=12000
HOST=0.0.0.0
AUTH_USERNAME=user
AUTH_PASSWORD=openskill
CUSTOM_DOMAIN=your-domain.com
EOF
```

#### Step 6: 启动服务器并提交

```bash
# 安装依赖
pip install flask python-dotenv

# 启动服务器
python server.py &

# 首次提交
git add .
git commit -m "[$(date +%Y-%m-%d)] init: 初始化 Build with Public 写作体系"
```

---

### Phase 2: 写作需求分析（含完整性检查）

#### 需求完整性评估

在正式开始前，先评估用户提供的信息是否足够。检查以下**必填项**：

| 检查项 | 状态 | 追问问题 |
|--------|:----:|----------|
| **写作主题** | ☐ | "您想写什么技术话题？" |
| **目标读者** | ☐ | "读者是什么水平？（初学者/进阶/专家）" |
| **核心目标** | ☐ | "文章主要目的是？（教学/分享/解决问题）" |
| **预期篇幅** | ☐ | "希望多长？（短文/中篇/长文）" |
| **技术背景** | ☐ | "涉及哪些技术栈/工具/框架？" |
| **个人经验** | ☐ | "您在这个领域的实战经验如何？" |
| **参考案例** | ☐ | "有参考的类似文章吗？" |
| **截止时间** | ☐ | "期望什么时候发布？" |

**判定标准**：
- ✅ **信息充足**（≥6项明确）→ 进入 Phase 3
- ⚠️ **部分缺失**（4-5项明确）→ 询问缺失项，可继续
- ❌ **严重不足**（≤3项明确）→ **必须补充后才能继续**

#### 追问策略

当条件宽泛时，使用**递进式追问**：

**第一轮（主题聚焦）**：
```
您提到的"{主题}"范围较广，为了提供精准的写作方案，
能否告诉我：
1. 具体的技术栈/工具是什么？
2. 您是想写教程类（How-to）还是经验分享类（Lesson-learned）？
3. 读者的技术水平大概是？
```

**第二轮（深度挖掘）**：
```
感谢您的补充！为了方案更贴合，还想了解：
1. 您在这个领域有多少实战经验？（影响文章可信度定位）
2. 有没有特别想强调的技术难点或踩坑经历？
3. 期望读者读完能收获什么？（学会技能/避坑/了解原理）
```

**第三轮（可行性确认）**：
```
最后确认两个细节：
1. 文章大概希望多长？（影响结构深度）
2. 期望什么时候发布？（影响迭代次数）
```

#### 最低可执行条件（MVP）

即使信息不完整，以下**4项必须有**，否则无法开始：

1. ✅ **写作主题** - 写什么？
2. ✅ **目标读者** - 给谁看？
3. ✅ **核心目标** - 想达成什么？
4. ✅ **技术背景** - 涉及什么技术？

**如果缺少以上任意一项，必须追问补充。**

#### Step 1: 深度理解写作主题

在信息充足后，进一步澄清：

**主题维度分析**：
- **技术深度**：原理层 / 应用层 / 工具层？
- **时间维度**：历史回顾 / 现状分析 / 未来展望？
- **实践维度**：纯理论 / 带案例 / 完整教程？

**示例对比**：

| 宽泛输入 | 澄清后 |
|----------|--------|
| "写关于 AI 的文章" | "写如何用 OpenClaw 构建技能评估工具的具体教程" |
| "分享我的经验" | "分享从技能消费者到贡献者的 12 小时实战记录" |
| "介绍某个工具" | "介绍 skill-explorer：8 阶段技能评估框架的设计与实现" |

#### Step 2: 质量目标设定

与用户确认质量期望：

| 维度 | 选项 | 影响 |
|------|------|------|
| **技术深度** | 入门科普 / 进阶实践 / 深度原理 | 决定代码示例和架构图数量 |
| **原创程度** | 经验复现 / 方案整合 / 原创方法 | 决定研究时间和创新点提炼 |
| **实用程度** | 概念介绍 / 可跟随操作 / 生产可用 | 决定代码完整度和测试要求 |
| **传播目标** | 个人笔记 / 社区分享 / 专业发表 | 决定打磨程度和推广策略 |

---

### Phase 3: 多角度方案设计

基于写作主题，设计 3-5 个不同的写作角度。

#### 常见角度模板

| 角度 | 适用场景 | 特点 |
|------|---------|------|
| **教程型** | 教读者如何做 | 步骤清晰，可复制 |
| **问题解决型** | 解决特定痛点 | 问题导向，实用性强 |
| **深度剖析型** | 深入技术原理 | 技术深度，专业性强 |
| **实战经验型** | 分享真实案例 | 真实可信，有借鉴价值 |
| **对比评测型** | 对比多种方案 | 客观中立，帮助决策 |
| **趋势展望型** | 分析发展方向 | 前瞻性强，引发思考 |

#### 输出大纲文件

为每个角度创建大纲文件：

```bash
# 文件命名格式：{date}-{topic}-{angle}-outline.md
# 例如：
# 2026-03-02-skill-explorer-tutorial-outline.md
# 2026-03-02-skill-explorer-case-study-outline.md
```

**大纲内容结构**：
```markdown
# 大纲选项：{角度名称}

**日期**: YYYY-MM-DD
**标题**: 《文章标题》

## 核心观点
- 一句话概括本文核心论点

## 大纲结构
1. **引言**: 引出问题和价值
2. **第一部分**: ...
3. **第二部分**: ...
4. **结论**: 总结和展望

## 目标读者
- 读者画像描述

## 预计字数
- XXXX 字

## 关键词
关键词1, 关键词2, ...
```

#### 保存到 draft 目录

```bash
# 示例
draft/
├── 2026-03-02-skill-explorer-tutorial-outline.md
├── 2026-03-02-skill-explorer-case-study-outline.md
├── 2026-03-02-skill-explorer-methodology-outline.md
└── ...
```

---

### Phase 4: 用户选择与确认

呈现方案供用户选择：

```markdown
## 可选写作方案

### 方案 1: 教程型 ⭐推荐
- **标题**: 《从零开始：如何构建 XXXX》
- **特点**: 步骤清晰，小白友好
- **预计**: 4000字，25分钟阅读

### 方案 2: 实战经验型
- **标题**: 《我如何在一周内解决 XXXX》
- **特点**: 真实案例，避坑指南
- **预计**: 3500字，20分钟阅读

### 方案 3: 深度剖析型
- **标题**: 《XXXX 背后的架构设计》
- **特点**: 技术深度，原理剖析
- **预计**: 5000字，30分钟阅读

**请选择**（或要求调整）: ...
```

用户确认后，进入写作阶段。

---

### Phase 5: 文章撰写

#### 版本编号规则

采用语义化版本：

| 版本 | 含义 | 示例 |
|------|------|------|
| v1.0 | 初稿完成 | 基础内容完整 |
| v1.1 | 小幅修订 | 修正错误、优化表达 |
| v2.0 | 重大改版 | 结构调整、角度改变 |
| v2.1 | 基于v2.0微调 | 细节优化 |

#### 文件命名规范

```
{date}-{topic}-{slug}-v{version}.md

示例：
- 2026-03-02-skill-explorer-tutorial-v1.md
- 2026-03-02-skill-explorer-tutorial-v2.md
- 2026-03-02-skill-explorer-tutorial-v2.1.md
```

#### 写作流程

1. **创建 v1.0**
   ```bash
   # 基于选定的大纲创建初稿
   cat > articles/2026-03-02-topic-v1.md << 'EOF'
   # 文章内容
   EOF
   ```

2. **迭代优化**
   - 根据反馈创建 v1.1, v1.2...
   - 重大调整时升级到 v2.0

3. **每次修改后提交 Git**
   ```bash
   git add articles/
   git commit -m "[YYYY-MM-DD] v{X.Y}: 修改描述"
   ```

---

### Phase 6: 链接分享

文章完成后，提供访问链接：

```
文章链接：http://your-domain.com:12000/articles/2026-03-02-topic-v2.md
```

**分享时的默认信息**：
- 访问需输入密码（默认 openskill）
- 浏览器会记住密码
- 支持 Markdown 文件直接下载

---

## 最佳实践

### Git 提交规范

```bash
# 格式
[日期] 类型: 描述

- 变更详情1
- 变更详情2

# 类型
feat:    新文章/新功能
fix:     错误修正
docs:    文档更新
refactor: 重构/重命名
style:   格式调整
```

### 版本控制策略

- **小改动**（错别字、格式）: 不升级版本号，直接覆盖
- **内容优化**（段落调整、补充细节）: v1.0 → v1.1
- **结构调整**（大纲变化、角度转换）: v1.x → v2.0

### 目录维护

定期清理：
- `draft/` 中已废弃的大纲
- `articles/` 中过时的版本（保留最新 2-3 个版本）

---

## Output Templates

### 方案选择展示模板

```markdown
## 📝 写作方案选项

基于您的需求「{主题}」，我设计了 {N} 个写作角度：

### 方案 1: {角度名称}
**标题**: 《{建议标题}》
**核心观点**: {一句话概括}
**适合读者**: {读者画像}
**预计篇幅**: {字数} 字
**大纲要点**:
1. {要点1}
2. {要点2}
3. {要点3}

---

[方案 2, 3 ...]

## 请选择

请回复数字（1/{N}）选择方案，或告诉我需要调整的方向。
```

### 文章完成通知模板

```markdown
✅ 文章已完成！

**版本**: v{X.Y}
**文件**: articles/{filename}
**字数**: {N} 字
**访问链接**: http://{domain}:12000/articles/{filename}

**访问密码**: openskill

**版本历史**:
- v1.0: 初稿完成
- v1.1: 优化表达
- v2.0: 结构调整
- v{X.Y}: 当前版本

如需修改，请告知具体要求。
```

---

## Examples

### Example 1: Skill Explorer 文章系列

**项目**: codewithriver  
**主题**: skill-explorer 技能开发  
**产出**:
- draft/2026-03-02-skill-explorer-tutorial-outline.md
- draft/2026-03-02-skill-explorer-case-study-outline.md
- draft/2026-03-02-skill-explorer-methodology-outline.md
- articles/2026-03-02-skill-explorer-tutorial-v1.md
- articles/2026-03-02-skill-explorer-tutorial-v2.md
- articles/2026-03-02-skill-explorer-tutorial-v2.1.md
- articles/2026-03-02-skill-explorer-tutorial-v2.2.md

---

## Related Skills

- **skill-explorer**: 评估和选择 ClawHub 技能
- **tweet-writer**: 撰写推广文案
- **marketing-mode**: 制定内容营销策略

---

## Tips

1. **保持迭代**: 好文章是改出来的，不要害怕版本号增加
2. **及时提交**: 每次有意义的改动都提交到 Git
3. **多角度尝试**: 同一主题的不同角度可能吸引不同读者
4. **链接分享**: 养成用链接而非发全文的习惯，培养读者访问习惯

---

*Build with Public - 让你的技术成长被看见*


---

## Appendix B: 小红书平台适配指南

### 为什么需要平台适配

同一技术内容可以发布到多个平台，但每个平台的用户习惯和内容形式不同。

**平台对比**:
- 技术博客: 专业开发者，深度长文，3000-8000字
- 小红书: 年轻学习者，轻松短笔记，500-1500字
- Twitter/X: 技术从业者，观点线程，280字/条
- 公众号: 泛技术人群，故事化文章，2000-5000字

### 小红书平台特点

**1. 标题公式**:
[表情] + [痛点/疑问] + [解决方案] + [表情] + [价值点]

示例:
- 🦞 OpenClaw技能太多挑花眼？我用这个秘籍8步搞定！✨
- 🔥 程序员必看！OpenClaw技能探索神器，告别踩坑！
- 💡 一天时间，从小白到发布第一个OpenClaw技能！

**2. 内容结构模板**:
- 开篇痛点（黄金3秒）
- 解决方案引入
- 步骤拆解+emoji编号
- 成果展示
- 福利时间
- 互动引导

**3. 语言风格转换**:
- AI Agent → OpenClaw / AI助手
- 开发者 → 宝子们 / 姐妹们
- 技能评估 → 技能挑选秘籍
- 系统化流程 → 8步搞定
- 经验分享 → 亲测有效

**4. 文件命名规范**:
```
{date}-{topic}-v{version}-xiaohongshu.md
```

### 多版本管理最佳实践

1. 先确定基础版本（v1.0/v2.0）技术博客版
2. 再衍生平台版本：cp topic-v2.md topic-v2-xiaohongshu.md
3. 按小红书模板改写内容和风格
4. 独立迭代：技术博客版和小红书版分别管理

### 快速转换技巧

使用 sed 命令批量替换术语:
```bash
sed -i 's/AI Agent/OpenClaw/g' article-xiaohongshu.md
sed -i 's/开发者/宝子们/g' article-xiaohongshu.md
```

---

### Phase 7: 版本管理与迭代

#### Git 工作流

每次文件变更后执行：
```bash
git add -A
git commit -m "[YYYY-MM-DD] vX.Y: 修改描述"
```

#### 版本控制策略

| 改动类型 | 版本变化 | 示例 |
|---------|---------|------|
| 错别字、格式 | 不升级 | 直接覆盖 |
| 段落优化 | v1.0 → v1.1 | 小幅修订 |
| 结构调整 | v1.x → v2.0 | 重大改版 |

#### 多版本并行管理

技术博客版和小红书版独立迭代：
```
articles/
├── topic-v2.md              # 技术博客版
├── topic-v2.1.md            # 技术博客版迭代
├── topic-v2-xiaohongshu.md  # 小红书版
└── topic-v2.1-xiaohongshu.md # 小红书版迭代
```

---

### Phase 8: 发布与推广

#### 发布前检查清单

- [ ] 内容完整，无错别字
- [ ] 代码测试通过
- [ ] 链接可正常访问
- [ ] Git 已提交

#### 推广渠道

1. **技术社区**：V2EX、掘金、知乎
2. **社交媒体**：Twitter、小红书、公众号
3. **开源平台**：GitHub、ClawHub

#### 互动策略

- 文章结尾提出互动问题
- 回复评论，建立连接
- 收集反馈，持续迭代

---

*Build with Public - 让你的技术成长被看见*