---
name: build-hook
description: >-
  Build a Uniswap V4 hook. Use when user wants to create a custom V4 hook
  contract. Generates Solidity code, Foundry tests, mines CREATE2 address
  for hook flags, and produces deployment scripts. Handles the full hook
  development lifecycle.
allowed-tools: >-
  Read, Write, Edit, Glob, Grep,
  Bash(forge:*), Bash(npm:*), Bash(git:*),
  Task(subagent_type:hook-builder),
  mcp__uniswap__get_supported_chains
model: opus
---

# Build Hook

## Overview

Builds a complete Uniswap V4 hook by delegating to the `hook-builder` agent. Handles the full development lifecycle: understanding requirements, determining hook flags, generating Solidity contracts, generating Foundry tests, mining a CREATE2 address with correct flag bits, and producing deployment scripts. Returns production-ready code artifacts written directly to the project.

## When to Use

Activate when the user asks:

- "Build a V4 hook"
- "Create a limit order hook"
- "Build a dynamic fee hook"
- "Create a TWAMM hook"
- "Custom hook for V4"
- "Hook that charges higher fees during volatility"
- "Build a hook that distributes LP fees to stakers"
- "Create a hook with oracle integration"

## Parameters

| Parameter | Required | Default | Description |
| --- | --- | --- | --- |
| behavior | Yes | -- | Hook behavior description (e.g., "limit orders", "dynamic fees", "TWAMM", "oracle-based pricing") |
| callbacks | No | Auto-detect | Specific V4 callbacks if the user knows them (e.g., "beforeSwap, afterSwap") |
| constraints | No | -- | Gas budget, security requirements, or specific design constraints |
| chain | No | ethereum | Target chain for deployment (affects PoolManager address) |

## Workflow

1. **Extract parameters** from the user's request: identify the hook behavior, any explicitly mentioned callbacks, constraints, and target chain.

2. **Delegate to hook-builder**: Invoke `Task(subagent_type:hook-builder)` with the full context. The hook-builder agent will:
   - Understand the requirements and determine which callbacks are needed
   - Map callbacks to hook flags and validate the flag combination
   - Generate a Solidity contract extending BaseHook with proper NatSpec
   - Generate comprehensive Foundry tests (unit, integration, edge cases, gas snapshots)
   - Mine a CREATE2 salt that produces an address encoding the required flags
   - Produce a deployment script with verification steps

3. **Present results** to the user with a summary covering:
   - Files written (contract path, test path, deployment script path)
   - Hook architecture explanation (what it does, how state flows)
   - Callbacks implemented and their flag bitmask
   - Gas estimates per callback (from Foundry test output)
   - Next steps for the developer (run tests, deploy to testnet, mainnet considerations)

## Output Format

Present a summary followed by the generated files:

```text
V4 Hook Built: LimitOrderHook

  Contract:   src/hooks/LimitOrderHook.sol (187 lines)
  Tests:      test/hooks/LimitOrderHook.t.sol (12 tests)
  Deployment: script/DeployLimitOrderHook.s.sol

  Callbacks: beforeSwap, afterSwap
  Flags:     0x00C0
  CREATE2:   Salt mined, address verified

  Gas Estimates:
    beforeSwap: ~45,000 gas
    afterSwap:  ~32,000 gas
    Total overhead per swap: ~77,000 gas

  Architecture:
    Orders are placed at specific ticks and stored in an on-chain order book.
    During beforeSwap, the hook checks for matching orders at the target tick.
    Matched orders are filled atomically within the same transaction.

  Next Steps:
    1. Run tests: forge test --match-contract LimitOrderHookTest
    2. Deploy to testnet: forge script script/DeployLimitOrderHook.s.sol --rpc-url sepolia
    3. Verify on Etherscan: forge verify-contract <address> LimitOrderHook
```

## Important Notes

- This skill delegates entirely to the `hook-builder` agent -- it does not call MCP tools directly.
- The hook-builder generates production-quality Solidity code with reentrancy protection and access control.
- CREATE2 address mining ensures the deployed address encodes the correct hook flags in its leading bytes (required by V4 PoolManager).
- Foundry must be installed for test generation and compilation. If not found, the skill will provide installation instructions.
- Generated code uses Solidity ^0.8.26 and imports from `@uniswap/v4-core` and `@uniswap/v4-periphery`.

## Error Handling

| Error | User-Facing Message | Suggested Action |
| --- | --- | --- |
| `INVALID_CALLBACK_COMBINATION` | "The requested behavior requires conflicting callbacks." | Simplify hook behavior or split into multiple hooks |
| `CREATE2_MINING_TIMEOUT` | "Could not mine a valid CREATE2 address within time limit." | Increase mining time limit or reduce required flags |
| `FORGE_NOT_INSTALLED` | "Foundry (forge) is required but not installed." | Install: `curl -L https://foundry.paradigm.xyz \| bash && foundryup` |
| `VAGUE_REQUIREMENTS` | "Need more detail about the desired hook behavior." | Describe specific behavior (e.g., "limit orders that execute at tick boundaries") |
| `COMPILATION_ERROR` | "Generated contract has compilation errors." | Review error output and adjust requirements |
