#!/usr/bin/env python3
from __future__ import annotations

import argparse
import base64
import hashlib
import hmac
import json
import os
import signal
import random
import time
import urllib.parse
import urllib.request
from dataclasses import dataclass
from pathlib import Path
from typing import Any
import html as _html


API_BASE_DEFAULT = "https://api.bricklink.com/api/store/v1"


def _find_workspace_root() -> Path:
    """Walk up from script location to find workspace root (parent of 'skills/')."""
    env = os.environ.get("BRICKLINK_WORKSPACE")
    if env:
        return Path(env)
    
    # Prefer CWD if it looks like a workspace (handles symlinks correctly)
    cwd = Path.cwd()
    if (cwd / "skills").is_dir():
        return cwd

    d = Path(__file__).resolve().parent
    for _ in range(6):
        if (d / "skills").is_dir() and d != d.parent:
            return d
        d = d.parent
    return Path.cwd()

# Quiet broken pipe behavior when piping to `head` etc.
try:
    signal.signal(signal.SIGPIPE, signal.SIG_DFL)
except Exception:
    pass


# ------------------------- oauth helpers -------------------------

def _pct(s: Any) -> str:
    """RFC3986 percent-encode (OAuth 1.0)."""
    return urllib.parse.quote(str(s), safe="~-._")


def _nonce(n: int = 16) -> str:
    alphabet = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
    return "".join(random.choice(alphabet) for _ in range(n))


def _normalize_base_url(url: str) -> str:
    """OAuth base string URI (scheme://host[:port]/path) without query/fragment."""
    u = urllib.parse.urlsplit(url)
    scheme = u.scheme.lower()
    host = (u.hostname or "").lower()
    port = u.port

    # Only include port if non-default
    include_port = port and not ((scheme == "http" and port == 80) or (scheme == "https" and port == 443))
    netloc = host
    if include_port:
        netloc = f"{host}:{port}"

    path = u.path or "/"
    return f"{scheme}://{netloc}{path}"


def _parse_query_params(url: str) -> list[tuple[str, str]]:
    u = urllib.parse.urlsplit(url)
    pairs = urllib.parse.parse_qsl(u.query, keep_blank_values=True)
    return [(str(k), str(v)) for k, v in pairs]


def _oauth_signature(
    method: str,
    url: str,
    query_params: list[tuple[str, str]],
    oauth_params: dict[str, str],
    consumer_secret: str,
    token_secret: str,
) -> str:
    # OAuth parameter normalization includes BOTH query params and oauth params
    all_params: list[tuple[str, str]] = []
    all_params.extend(query_params)
    all_params.extend([(k, v) for k, v in oauth_params.items() if k != "oauth_signature"])

    # Percent-encode keys/values, sort
    enc = [(_pct(k), _pct(v)) for k, v in all_params]
    enc.sort(key=lambda kv: (kv[0], kv[1]))

    param_str = "&".join([f"{k}={v}" for k, v in enc])
    base_elems = [
        method.upper(),
        _pct(_normalize_base_url(url)),
        _pct(param_str),
    ]
    base_string = "&".join(base_elems)

    key = f"{_pct(consumer_secret)}&{_pct(token_secret)}"
    digest = hmac.new(key.encode("utf-8"), base_string.encode("utf-8"), hashlib.sha1).digest()
    return base64.b64encode(digest).decode("ascii")


def _auth_header(oauth_params: dict[str, str]) -> str:
    # Authorization: OAuth key="value", ... (values are percent-encoded)
    items = []
    for k in sorted(oauth_params.keys()):
        if not k.startswith("oauth_"):
            continue
        items.append(f'{_pct(k)}="{_pct(oauth_params[k])}"')
    return "OAuth " + ", ".join(items)


# ------------------------- config -------------------------

@dataclass
class Creds:
    consumer_key: str
    consumer_secret: str
    token_value: str
    token_secret: str


def _env(name: str) -> str | None:
    v = os.environ.get(name)
    if v is None:
        return None
    v = v.strip()
    return v or None


def _load_creds_from_json(path: str) -> Creds | None:
    try:
        obj = json.loads(open(path, "r", encoding="utf-8").read())
    except Exception:
        return None

    if not isinstance(obj, dict):
        return None

    # Accept either flat keys or nested under "oauth".
    src = obj.get("oauth") if isinstance(obj.get("oauth"), dict) else obj

    ck = src.get("consumer_key") or src.get("consumerKey")
    cs = src.get("consumer_secret") or src.get("consumerSecret")
    tv = src.get("token_value") or src.get("tokenValue")
    ts = src.get("token_secret") or src.get("tokenSecret")

    if all(isinstance(x, str) and x.strip() for x in (ck, cs, tv, ts)):
        return Creds(ck.strip(), cs.strip(), tv.strip(), ts.strip())
    return None


def load_creds(args) -> Creds:
    # 1) Config JSON from workspace/bricklink/config.json only
    cfg = str(_find_workspace_root() / "bricklink" / "config.json")
    if os.path.exists(cfg):
        c = _load_creds_from_json(cfg)
        if c:
            return c

    # 2) Env vars
    ck = _env("BRICKLINK_CONSUMER_KEY")
    cs = _env("BRICKLINK_CONSUMER_SECRET")
    tv = _env("BRICKLINK_TOKEN_VALUE")
    ts = _env("BRICKLINK_TOKEN_SECRET")

    missing = [k for k, v in {
        "BRICKLINK_CONSUMER_KEY": ck,
        "BRICKLINK_CONSUMER_SECRET": cs,
        "BRICKLINK_TOKEN_VALUE": tv,
        "BRICKLINK_TOKEN_SECRET": ts,
    }.items() if not v]

    if missing:
        raise SystemExit(
            "Missing BrickLink OAuth creds: " + ", ".join(missing) +
            "\nTip: set env vars or place config.json in workspace/bricklink/"
        )

    return Creds(ck, cs, tv, ts)  # type: ignore[arg-type]


# ------------------------- http -------------------------

def api_call(
    creds: Creds,
    method: str,
    url: str,
    *,
    body_json: Any | None = None,
    extra_headers: dict[str, str] | None = None,
    timeout: int = 30,
) -> Any:
    method_u = method.upper()

    query_params = _parse_query_params(url)

    oauth_params: dict[str, str] = {
        "oauth_consumer_key": creds.consumer_key,
        "oauth_token": creds.token_value,
        "oauth_nonce": _nonce(),
        "oauth_timestamp": str(int(time.time())),
        "oauth_signature_method": "HMAC-SHA1",
        "oauth_version": "1.0",
    }

    sig = _oauth_signature(method_u, url, query_params, oauth_params, creds.consumer_secret, creds.token_secret)
    oauth_params["oauth_signature"] = sig

    headers = {
        "Accept": "application/json",
        "Authorization": _auth_header(oauth_params),
        "User-Agent": "moltbot/bricklink",
    }
    if extra_headers:
        headers.update(extra_headers)

    data = None
    if body_json is not None:
        data = json.dumps(body_json).encode("utf-8")
        headers["Content-Type"] = "application/json"

    req = urllib.request.Request(url, data=data, headers=headers, method=method_u)

    try:
        with urllib.request.urlopen(req, timeout=timeout) as resp:
            raw = resp.read()
            ct = resp.headers.get("Content-Type", "")
    except urllib.error.HTTPError as e:
        raw = e.read() if hasattr(e, "read") else b""
        try:
            payload = json.loads(raw.decode("utf-8", errors="ignore")) if raw else {"error": str(e)}
        except Exception:
            payload = {"error": str(e), "body": raw.decode("utf-8", errors="ignore")}
        raise SystemExit(json.dumps(payload, ensure_ascii=False, indent=2))

    if b"" == raw:
        return None

    # BrickLink returns JSON
    try:
        return json.loads(raw.decode("utf-8"))
    except Exception:
        # Fallback
        return {"contentType": ct, "raw": raw.decode("utf-8", errors="ignore")}


# ------------------------- rendering -------------------------

def _money(amount: Any, currency: str | None) -> str:
    if amount is None:
        return ""
    s = str(amount)
    cur = (currency or "").strip()
    if cur:
        return f"{cur} {s}"
    return s


def _unescape_entities(s: Any) -> str:
    # BrickLink sometimes returns HTML entities inside fields (e.g. "&#40;" for "(")
    return _html.unescape("" if s is None else str(s))


def _html_escape(s: Any) -> str:
    # Always unescape first, then escape for HTML output
    s2 = _unescape_entities(s)
    return (
        s2
        .replace("&", "&amp;")
        .replace("<", "&lt;")
        .replace(">", "&gt;")
        .replace('"', "&quot;")
    )


def _item_image_url(item_type: str, item_no: str, color_id: int | None) -> str | None:
    t = (item_type or "").upper()
    no = (item_no or "").strip()
    if not no:
        return None
    # This matches the classic BrickLink image pattern used by orderDetail.asp
    if t == "PART" and color_id:
        return f"https://img.bricklink.com/P/{int(color_id)}/{urllib.parse.quote(no)}.jpg"
    if t == "PART":
        return f"https://img.bricklink.com/P/0/{urllib.parse.quote(no)}.jpg"
    if t == "MINIFIG":
        return f"https://img.bricklink.com/M/{urllib.parse.quote(no)}.jpg"
    if t == "SET":
        return f"https://img.bricklink.com/S/{urllib.parse.quote(no)}.jpg"
    # Fallback
    return None


def _fetch_data_uri(url: str) -> str | None:
    try:
        req = urllib.request.Request(url, headers={"User-Agent": "moltbot/bricklink"})
        with urllib.request.urlopen(req, timeout=20) as r:
            data = r.read()
            ct = r.headers.get("Content-Type", "image/jpeg").split(";")[0].strip() or "image/jpeg"
        b64 = base64.b64encode(data).decode("ascii")
        return f"data:{ct};base64,{b64}"
    except Exception:
        return None


def render_order_detail_html(order: dict, batches: list[list[dict]], inline_images: bool = False) -> str:
    """Render a compact HTML view similar to orderDetail.asp.html.

    Includes:
    - Purchaser name/address/email/phone
    - Items table: Qty, Picture, Part #, Part Name, Comment

    If inline_images=True, images are embedded as data: URIs (works in sandboxed HTML viewers).
    """
    oid = order.get("order_id")
    status = order.get("status")
    buyer_username = order.get("buyer_name")
    buyer_email = order.get("buyer_email")
    seller = order.get("seller_name")
    store = order.get("store_name")
    date_ordered = order.get("date_ordered")

    payment = order.get("payment") if isinstance(order.get("payment"), dict) else {}
    cost = order.get("disp_cost") if isinstance(order.get("disp_cost"), dict) else (order.get("cost") if isinstance(order.get("cost"), dict) else {})

    shipping = order.get("shipping") if isinstance(order.get("shipping"), dict) else {}
    ship_addr = shipping.get("address") if isinstance(shipping.get("address"), dict) else {}
    ship_name = ship_addr.get("name") if isinstance(ship_addr.get("name"), dict) else {}

    buyer_full_name = ship_name.get("full") or buyer_username
    buyer_phone = ship_addr.get("phone_number")

    addr_lines = []
    for k in ("address1", "address2"):
        v = ship_addr.get(k)
        if isinstance(v, str) and v.strip():
            addr_lines.append(v.strip())
    city = ship_addr.get("city")
    postal = ship_addr.get("postal_code")
    state = ship_addr.get("state")
    country = ship_addr.get("country_code")

    city_line_parts = []
    if postal: city_line_parts.append(str(postal))
    if city: city_line_parts.append(str(city))
    if state: city_line_parts.append(str(state))
    if country: city_line_parts.append(str(country))
    if city_line_parts:
        addr_lines.append(" ".join(city_line_parts))

    cur = cost.get("currency_code") or payment.get("currency_code")

    # Flatten items
    flat_items: list[dict] = []
    for b in batches:
        for it in b:
            if isinstance(it, dict):
                flat_items.append(it)

    # Totals
    total_qty = 0
    for it in flat_items:
        try:
            total_qty += int(it.get("quantity") or 0)
        except Exception:
            pass

    rows = []
    for it in flat_items:
        item = it.get("item") if isinstance(it.get("item"), dict) else {}
        item_no = str(item.get("no") or "")
        item_name = str(item.get("name") or "")
        item_type = str(item.get("type") or "")
        color_id = it.get("color_id")
        color_name = it.get("color_name")

        q = it.get("quantity")
        comment = it.get("remarks") or it.get("description") or ""

        img = _item_image_url(item_type, item_no, int(color_id) if color_id is not None else None)
        img_html = ""
        if img:
            src = img
            if inline_images:
                data_uri = _fetch_data_uri(img)
                if data_uri:
                    src = data_uri
            img_html = f"<img src='{_html_escape(src)}' style='width:80px;height:60px;object-fit:contain'/>"

        # Part # column: show number, with color name under it
        part_cell = _html_escape(item_no)
        if color_name:
            part_cell += f"<br><span class='muted'>{_html_escape(color_name)}</span>"

        rows.append(
            "<tr>"
            f"<td class='right qty'>{_html_escape(q)}</td>"
            f"<td class='img'>{img_html}</td>"
            f"<td class='wrap'>{part_cell}</td>"
            f"<td class='wrap'>{_html_escape(item_name)}</td>"
            f"<td class='wrap muted'>{_html_escape(comment)}</td>"
            "</tr>"
        )

    return f"""<!doctype html>
<html><head><meta charset='utf-8'>
<title>Order {oid}</title>
<style>
body {{ font-family: -apple-system, BlinkMacSystemFont, Segoe UI, Roboto, Helvetica, Arial, sans-serif; margin: 20px; }}
h1 {{ margin: 0 0 8px 0; }}
h2 {{ margin: 16px 0 8px 0; }}
.muted {{ color: #666; font-size: 12px; }}
.box {{ border: 1px solid #ddd; border-radius: 8px; padding: 12px; margin: 12px 0; }}
.grid {{ display: grid; grid-template-columns: 160px 1fr; gap: 4px 12px; }}
label {{ color: #666; font-size: 12px; }}
table {{ width: 100%; border-collapse: collapse; }}
th, td {{ border-bottom: 1px solid #eee; padding: 8px; vertical-align: top; }}
th {{ text-align: left; background: #f7f7f7; }}
.right {{ text-align: right; }}
.wrap {{ word-break: break-word; }}
td.img {{ width: 90px; }}
td.qty {{ width: 60px; }}
</style>
</head>
<body>
  <h1>Order #{_html_escape(oid)}</h1>
  <div class='muted'>Status: <strong>{_html_escape(status)}</strong> · Ordered: {_html_escape(date_ordered)}</div>

  <div class='box'>
    <div class='grid'>
      <label>Seller</label><div>{_html_escape(seller)} ({_html_escape(store)})</div>
      <label>Buyer (username)</label><div>{_html_escape(buyer_username)}</div>
      <label>Name</label><div>{_html_escape(buyer_full_name)}</div>
      <label>Address</label><div>{'<br>'.join([_html_escape(x) for x in addr_lines])}</div>
      <label>Email</label><div>{_html_escape(buyer_email)}</div>
      <label>Phone</label><div>{_html_escape(buyer_phone)}</div>
    </div>
  </div>

  <div class='box'>
    <div class='grid'>
      <label>Payment</label><div>{_html_escape(payment.get('method'))} · {_html_escape(payment.get('status'))}</div>
      <label>Totals</label><div>items {_html_escape(order.get('total_count'))} · lots {_html_escape(order.get('unique_count'))} · grand total {_html_escape(cost.get('grand_total') or cost.get('grandtotal'))} {_html_escape(cur)}</div>
    </div>
  </div>

  <h2>Items</h2>
  <table>
    <thead>
      <tr><th class='right'>Qty</th><th>Picture</th><th>Part #</th><th>Part name</th><th>Comment</th></tr>
    </thead>
    <tbody>
      {''.join(rows) if rows else '<tr><td colspan=5 class="muted">(no items)</td></tr>'}
    </tbody>
  </table>
</body></html>"""


_INVENTORY_KEYS = {
    "item", "color_id", "color_name", "quantity", "new_or_used",
    "completeness", "unit_price", "bind_id", "description", "remarks",
    "bulk", "is_retain", "is_stock_room", "stock_room_id",
    "my_cost", "sale_rate", "tier_quantity1", "tier_quantity2", "tier_quantity3",
    "tier_price1", "tier_price2", "tier_price3",
}


def _load_batch_file(path_str: str) -> list[dict[str, Any]]:
    """Load and validate a JSON batch file for inventory operations."""
    p = Path(path_str).resolve()

    # Path restriction: workspace + /tmp only
    allowed = [Path("/tmp").resolve()]
    cwd = Path.cwd()
    if (cwd / "skills").is_dir():
        allowed.append(cwd.resolve())
    ws = os.environ.get("OPENCLAW_WORKSPACE")
    if ws:
        allowed.append(Path(ws).resolve())
    if not any(str(p).startswith(str(a) + "/") or p == a for a in allowed):
        raise SystemExit(f"Refusing to read '{path_str}' — must be inside workspace or /tmp")

    # Must be a .json file
    if p.suffix.lower() != ".json":
        raise SystemExit(f"Batch file must be a .json file, got: {p.name}")

    if not p.is_file():
        raise SystemExit(f"Batch file not found: {path_str}")

    try:
        with open(p, "r", encoding="utf-8") as f:
            data = json.load(f)
    except (json.JSONDecodeError, UnicodeDecodeError) as e:
        raise SystemExit(f"Invalid JSON in batch file: {e}")

    if not isinstance(data, list):
        raise SystemExit("Batch file must contain a JSON array of inventory objects")
    if not data:
        raise SystemExit("Batch file contains an empty array")

    for idx, entry in enumerate(data):
        if not isinstance(entry, dict):
            raise SystemExit(f"Batch item [{idx}] is not a JSON object")
        unknown = set(entry.keys()) - _INVENTORY_KEYS
        if unknown:
            raise SystemExit(f"Batch item [{idx}] has unknown keys: {', '.join(sorted(unknown))}")

    return data


def _apply_inventory_flags(body: dict[str, Any], args, *, include_item: bool = True) -> dict[str, Any]:
    if include_item:
        item: dict[str, Any] = body.get("item") if isinstance(body.get("item"), dict) else {}
        if args.item_type is not None:
            item["type"] = str(args.item_type)
        if args.item_no is not None:
            item["no"] = str(args.item_no)
        if item:
            body["item"] = item

    if args.color_id is not None:
        body["color_id"] = int(args.color_id)
    if args.quantity is not None:
        body["quantity"] = int(args.quantity)
    if args.unit_price is not None:
        body["unit_price"] = str(args.unit_price)
    if args.new_or_used is not None:
        body["new_or_used"] = str(args.new_or_used)
    if args.completeness is not None:
        body["completeness"] = str(args.completeness)
    if args.description is not None:
        body["description"] = str(args.description)
    if args.remarks is not None:
        body["remarks"] = str(args.remarks)
    if args.bulk is not None:
        body["bulk"] = bool(args.bulk)
    if args.is_retain is not None:
        body["is_retain"] = bool(args.is_retain)
    if args.is_stock_room is not None:
        body["is_stock_room"] = bool(args.is_stock_room)
    if args.stock_room_id is not None:
        body["stock_room_id"] = int(args.stock_room_id)
    if args.my_cost is not None:
        body["my_cost"] = str(args.my_cost)
    if args.sale_rate is not None:
        body["sale_rate"] = str(args.sale_rate)

    return body


def _validate_inventory_required(body: dict[str, Any], *, index: int | None = None) -> None:
    missing: list[str] = []
    item = body.get("item")
    if not isinstance(item, dict):
        missing.append("item")
        item = {}
    if "type" not in item:
        missing.append("item.type")
    if "no" not in item:
        missing.append("item.no")
    if "color_id" not in body:
        missing.append("color_id")
    if "quantity" not in body:
        missing.append("quantity")
    if "unit_price" not in body:
        missing.append("unit_price")
    if "new_or_used" not in body:
        missing.append("new_or_used")

    if missing:
        prefix = f"Inventory[{index}] missing required fields: " if index is not None else "Inventory missing required fields: "
        raise SystemExit(prefix + ", ".join(missing))


def _add_inventory_common_args(parser: argparse.ArgumentParser, *, include_item: bool) -> None:
    if include_item:
        parser.add_argument("--item-type", default=None, help="Item type (PART, SET, MINIFIG, ...)")
        parser.add_argument("--item-no", default=None, help="Item number (e.g. 3001, 7644-1)")
    parser.add_argument("--color-id", type=int, default=None)
    parser.add_argument("--quantity", type=int, default=None)
    parser.add_argument("--unit-price", default=None)
    parser.add_argument("--new-or-used", choices=["N", "U"], default=None)
    parser.add_argument("--completeness", default=None, help="C, B, S, or M (for sets/minifigs)")
    parser.add_argument("--description", default=None)
    parser.add_argument("--remarks", default=None)
    parser.add_argument("--bulk", default=None, choices=["true", "false"])
    parser.add_argument("--is-retain", default=None, choices=["true", "false"])
    parser.add_argument("--is-stock-room", default=None, choices=["true", "false"])
    parser.add_argument("--stock-room-id", type=int, default=None)
    parser.add_argument("--my-cost", default=None)
    parser.add_argument("--sale-rate", default=None)

# ------------------------- commands -------------------------

def _compose_include_exclude(raw: str | None, include: list[str] | None, exclude: list[str] | None) -> str | None:
    if raw and str(raw).strip():
        return str(raw).strip()
    inc = include or []
    exc = exclude or []
    parts: list[str] = []
    parts.extend([str(s).strip() for s in inc if str(s).strip()])
    parts.extend([f"-{str(s).strip()}" for s in exc if str(s).strip()])
    if parts:
        return ",".join(parts)
    return None


def cmd_get_orders(args) -> int:
    creds = load_creds(args)
    base = API_BASE_DEFAULT

    params: dict[str, str] = {}
    if args.direction:
        params["direction"] = args.direction

    status = _compose_include_exclude(getattr(args, "status", None), getattr(args, "include_status", None), getattr(args, "exclude_status", None))
    if status:
        params["status"] = status

    if args.filed is not None:
        params["filed"] = "true" if args.filed else "false"

    qs = urllib.parse.urlencode(params) if params else ""
    url = f"{base}/orders" + (f"?{qs}" if qs else "")

    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_order(args) -> int:
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/orders/{int(args.order_id)}"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_inventories(args) -> int:
    """GET /inventories with optional filters."""
    creds = load_creds(args)
    base = API_BASE_DEFAULT

    params: dict[str, str] = {}

    item_type = _compose_include_exclude(getattr(args, "item_type", None), getattr(args, "include_item_type", None), getattr(args, "exclude_item_type", None))
    if item_type:
        params["item_type"] = item_type

    status = _compose_include_exclude(getattr(args, "status", None), getattr(args, "include_status", None), getattr(args, "exclude_status", None))
    if status:
        params["status"] = status

    category_id = _compose_include_exclude(getattr(args, "category_id", None), getattr(args, "include_category_id", None), getattr(args, "exclude_category_id", None))
    if category_id:
        params["category_id"] = category_id

    color_id = _compose_include_exclude(getattr(args, "color_id", None), getattr(args, "include_color_id", None), getattr(args, "exclude_color_id", None))
    if color_id:
        params["color_id"] = color_id

    qs = urllib.parse.urlencode(params) if params else ""
    url = f"{base}/inventories" + (f"?{qs}" if qs else "")

    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_inventory(args) -> int:
    """GET /inventories/{inventory_id}"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/inventories/{int(args.inventory_id)}"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def _item_type_path(t: str) -> str:
    # API examples use lowercase in the path
    return (t or "").strip().lower()


def cmd_get_item(args) -> int:
    """GET /items/{type}/{no}"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/items/{_item_type_path(args.type)}/{urllib.parse.quote(str(args.no))}"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_supersets(args) -> int:
    """GET /items/{type}/{no}/supersets"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT

    params: dict[str, str] = {}
    if args.color_id is not None:
        params["color_id"] = str(int(args.color_id))

    qs = urllib.parse.urlencode(params) if params else ""
    url = f"{base}/items/{_item_type_path(args.type)}/{urllib.parse.quote(str(args.no))}/supersets" + (f"?{qs}" if qs else "")

    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_subsets(args) -> int:
    """GET /items/{type}/{no}/subsets"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT

    params: dict[str, str] = {}
    if args.color_id is not None:
        params["color_id"] = str(int(args.color_id))
    if args.box is not None:
        params["box"] = "true" if args.box else "false"
    if args.instruction is not None:
        params["instruction"] = "true" if args.instruction else "false"
    if args.break_minifigs is not None:
        params["break_minifigs"] = "true" if args.break_minifigs else "false"
    if args.break_subsets is not None:
        params["break_subsets"] = "true" if args.break_subsets else "false"

    qs = urllib.parse.urlencode(params) if params else ""
    url = f"{base}/items/{_item_type_path(args.type)}/{urllib.parse.quote(str(args.no))}/subsets" + (f"?{qs}" if qs else "")

    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_price_guide(args) -> int:
    """GET /items/{type}/{no}/price"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT

    params: dict[str, str] = {}
    if args.color_id is not None:
        params["color_id"] = str(int(args.color_id))
    if args.guide_type:
        params["guide_type"] = str(args.guide_type)
    if args.new_or_used:
        params["new_or_used"] = str(args.new_or_used)
    if args.country_code:
        params["country_code"] = str(args.country_code)
    if args.region:
        params["region"] = str(args.region)
    if args.currency_code:
        params["currency_code"] = str(args.currency_code)
    if args.vat:
        params["vat"] = str(args.vat)

    qs = urllib.parse.urlencode(params) if params else ""
    url = f"{base}/items/{_item_type_path(args.type)}/{urllib.parse.quote(str(args.no))}/price" + (f"?{qs}" if qs else "")

    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_known_colors(args) -> int:
    """GET /items/{type}/{no}/colors"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/items/{_item_type_path(args.type)}/{urllib.parse.quote(str(args.no))}/colors"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_order_items(args) -> int:
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/orders/{int(args.order_id)}/items"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_order_messages(args) -> int:
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/orders/{int(args.order_id)}/messages"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_order_feedback(args) -> int:
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/orders/{int(args.order_id)}/feedback"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_feedback(args) -> int:
    """GET /feedback"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT

    params: dict[str, str] = {}
    if args.direction:
        params["direction"] = args.direction

    qs = urllib.parse.urlencode(params) if params else ""
    url = f"{base}/feedback" + (f"?{qs}" if qs else "")
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_feedback_item(args) -> int:
    """GET /feedback/{feedback_id}"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/feedback/{int(args.feedback_id)}"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_post_feedback(args) -> int:

    creds = load_creds(args)
    base = API_BASE_DEFAULT

    body: dict[str, Any] = {}
    if args.order_id is not None:
        body["order_id"] = int(args.order_id)
    if args.rating is not None:
        body["rating"] = int(args.rating)
    if args.comment is not None:
        body["comment"] = str(args.comment)

    if not body:
        raise SystemExit("Feedback body is empty. Provide --order-id/--rating/--comment.")
    if "order_id" not in body or "rating" not in body:
        raise SystemExit("Feedback requires order_id and rating (use --order-id and --rating).")

    url = f"{base}/feedback"
    data = api_call(creds, "POST", url, body_json=body)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_reply_feedback(args) -> int:

    creds = load_creds(args)
    base = API_BASE_DEFAULT

    body: dict[str, Any] = {}
    if args.reply is not None:
        body["reply"] = str(args.reply)

    if not body:
        raise SystemExit("Reply body is empty. Provide --reply.")
    if "reply" not in body:
        raise SystemExit("Reply requires reply text (use --reply).")

    fid = int(args.feedback_id)
    url = f"{base}/feedback/{fid}/reply"
    data = api_call(creds, "POST", url, body_json=body)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_notifications(args) -> int:
    """GET /notifications"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/notifications"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_categories(args) -> int:
    """GET /categories"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/categories"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_colors(args) -> int:
    """GET /colors"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/colors"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_color(args) -> int:
    """GET /colors/{color_id}"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/colors/{int(args.color_id)}"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_get_category(args) -> int:
    """GET /categories/{category_id}"""
    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/categories/{int(args.category_id)}"
    data = api_call(creds, "GET", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_create_inventory(args) -> int:

    creds = load_creds(args)
    base = API_BASE_DEFAULT
    url = f"{base}/inventories"

    if args.file:
        # Batch mode: post array from JSON file
        items = _load_batch_file(args.file)
        for idx, entry in enumerate(items):
            _validate_inventory_required(entry, index=idx)
        print(f"Creating {len(items)} inventory item(s)…", file=__import__('sys').stderr)
        data = api_call(creds, "POST", url, body_json=items)
    else:
        # Single mode: build from flags
        body: dict[str, Any] = {}
        body = _apply_inventory_flags(body, args, include_item=True)
        if not body:
            raise SystemExit("Provide item flags (--item-type, --item-no, etc.) or --file for batch.")
        _validate_inventory_required(body)
        data = api_call(creds, "POST", url, body_json=body)

    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_update_inventory(args) -> int:

    creds = load_creds(args)
    base = API_BASE_DEFAULT

    body: dict[str, Any] = {}
    body = _apply_inventory_flags(body, args, include_item=False)

    if not body:
        raise SystemExit("Inventory update body is empty. Provide flags (--quantity, --unit-price, etc.).")

    inv_id = int(args.inventory_id)
    url = f"{base}/inventories/{inv_id}"
    data = api_call(creds, "PUT", url, body_json=body)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_delete_inventory(args) -> int:

    creds = load_creds(args)
    base = API_BASE_DEFAULT
    inv_id = int(args.inventory_id)
    url = f"{base}/inventories/{inv_id}"
    data = api_call(creds, "DELETE", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_update_order(args) -> int:

    creds = load_creds(args)
    base = API_BASE_DEFAULT

    body: dict[str, Any] = {}

    # Apply flags
    if args.remarks is not None:
        body["remarks"] = args.remarks

    if args.is_filed is not None:
        body["is_filed"] = bool(args.is_filed)

    shipping: dict[str, Any] = body.get("shipping") if isinstance(body.get("shipping"), dict) else {}
    if args.shipping_date_shipped is not None:
        shipping["date_shipped"] = args.shipping_date_shipped
    if args.shipping_tracking_no is not None:
        shipping["tracking_no"] = args.shipping_tracking_no
    if args.shipping_tracking_link is not None:
        shipping["tracking_link"] = args.shipping_tracking_link
    if args.shipping_method_id is not None:
        shipping["method_id"] = int(args.shipping_method_id)
    if shipping:
        body["shipping"] = shipping

    cost: dict[str, Any] = body.get("cost") if isinstance(body.get("cost"), dict) else {}
    for key, val in (
        ("shipping", args.cost_shipping),
        ("insurance", args.cost_insurance),
        ("credit", args.cost_credit),
        ("etc1", args.cost_etc1),
        ("etc2", args.cost_etc2),
    ):
        if val is not None:
            cost[key] = val
    if cost:
        body["cost"] = cost

    oid = int(args.order_id)
    url = f"{base}/orders/{oid}"
    data = api_call(creds, "PUT", url, body_json=body)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_update_order_status(args) -> int:

    creds = load_creds(args)
    base = API_BASE_DEFAULT
    oid = int(args.order_id)
    url = f"{base}/orders/{oid}/status"
    body = {"field": "status", "value": args.status}
    data = api_call(creds, "PUT", url, body_json=body)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_update_payment_status(args) -> int:

    creds = load_creds(args)
    base = API_BASE_DEFAULT
    oid = int(args.order_id)
    url = f"{base}/orders/{oid}/payment_status"
    body = {"field": "payment_status", "value": args.payment_status}
    data = api_call(creds, "PUT", url, body_json=body)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_send_drive_thru(args) -> int:

    creds = load_creds(args)
    base = API_BASE_DEFAULT
    oid = int(args.order_id)

    params = {}
    if args.mail_me:
        params["mail_me"] = "true"

    qs = urllib.parse.urlencode(params) if params else ""
    url = f"{base}/orders/{oid}/drive_thru" + (f"?{qs}" if qs else "")

    data = api_call(creds, "POST", url)
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return 0


def cmd_order_detail_html(args) -> int:
    creds = load_creds(args)
    base = API_BASE_DEFAULT

    oid = int(args.order_id)
    order = api_call(creds, "GET", f"{base}/orders/{oid}")
    items = api_call(creds, "GET", f"{base}/orders/{oid}/items")

    order_data = order.get("data") if isinstance(order, dict) else None
    batches = items.get("data") if isinstance(items, dict) else None
    if not isinstance(order_data, dict) or not isinstance(batches, list):
        raise SystemExit("Unexpected API response shape")

    # batches is list[list[item]]
    norm_batches: list[list[dict]] = []
    for b in batches:
        if isinstance(b, list):
            norm_batches.append([it for it in b if isinstance(it, dict)])

    html = render_order_detail_html(order_data, norm_batches, inline_images=bool(getattr(args, "inline_images", False)))

    from _pathguard import safe_output_path

    raw_out = args.out or f"/tmp/bricklink_order_{oid}_detail.html"
    out_path = str(safe_output_path(raw_out))

    with open(out_path, "w", encoding="utf-8") as f:
        f.write(html)

    print(out_path)
    return 0


def main() -> int:
    ap = argparse.ArgumentParser(description="BrickLink Store API CLI (OAuth 1.0)")

    sub = ap.add_subparsers(dest="cmd", required=True)

    p = sub.add_parser("get-orders", help="GET /orders")
    p.add_argument("--direction", choices=["in", "out"], default=None)

    # Two ways to specify status filters:
    # 1) Raw passthrough: --status "pending,completed" or "-purged"
    # 2) Structured: --include-status pending --include-status completed --exclude-status purged
    p.add_argument("--status", default=None, help="Raw BrickLink status filter string (e.g. pending,completed or -purged)")
    p.add_argument("--include-status", action="append", default=None, help="Include status (repeatable)")
    p.add_argument("--exclude-status", action="append", default=None, help="Exclude status (repeatable)")

    p.add_argument("--filed", default=None, choices=["true", "false"], help="Filter filed/unfiled")
    p.set_defaults(func=cmd_get_orders)

    p = sub.add_parser("get-order", help="GET /orders/{order_id}")
    p.add_argument("order_id", type=int)
    p.set_defaults(func=cmd_get_order)

    p = sub.add_parser("get-order-items", help="GET /orders/{order_id}/items")
    p.add_argument("order_id", type=int)
    p.set_defaults(func=cmd_get_order_items)

    p = sub.add_parser("get-inventories", help="GET /inventories")
    # raw passthrough filters
    p.add_argument("--item-type", dest="item_type", default=None, help="Raw item_type filter (comma-separated, -exclude)")
    p.add_argument("--status", default=None, help="Raw status filter (comma-separated, -exclude)")
    p.add_argument("--category-id", dest="category_id", default=None, help="Raw category_id filter (comma-separated, -exclude)")
    p.add_argument("--color-id", dest="color_id", default=None, help="Raw color_id filter (comma-separated, -exclude)")
    # structured filters
    p.add_argument("--include-item-type", action="append", default=None)
    p.add_argument("--exclude-item-type", action="append", default=None)
    p.add_argument("--include-status", action="append", default=None)
    p.add_argument("--exclude-status", action="append", default=None)
    p.add_argument("--include-category-id", action="append", default=None)
    p.add_argument("--exclude-category-id", action="append", default=None)
    p.add_argument("--include-color-id", action="append", default=None)
    p.add_argument("--exclude-color-id", action="append", default=None)
    p.set_defaults(func=cmd_get_inventories)

    p = sub.add_parser("get-inventory", help="GET /inventories/{inventory_id}")
    p.add_argument("inventory_id", type=int)
    p.set_defaults(func=cmd_get_inventory)

    # Catalog items
    p = sub.add_parser("get-item", help="GET /items/{type}/{no}")
    p.add_argument("type", help="Item type (PART, SET, MINIFIG, ...) ")
    p.add_argument("no", help="Item number (e.g. 3001old, 7644-1)")
    p.set_defaults(func=cmd_get_item)

    p = sub.add_parser("get-supersets", help="GET /items/{type}/{no}/supersets")
    p.add_argument("type")
    p.add_argument("no")
    p.add_argument("--color-id", type=int, default=None)
    p.set_defaults(func=cmd_get_supersets)

    p = sub.add_parser("get-subsets", help="GET /items/{type}/{no}/subsets")
    p.add_argument("type")
    p.add_argument("no")
    p.add_argument("--color-id", type=int, default=None)
    p.add_argument("--box", default=None, choices=["true", "false"], help="Include original box (for sets)")
    p.add_argument("--instruction", default=None, choices=["true", "false"], help="Include instruction (for sets)")
    p.add_argument("--break-minifigs", default=None, choices=["true", "false"], help="Break down minifigs into parts")
    p.add_argument("--break-subsets", default=None, choices=["true", "false"], help="Break down sets-in-set")
    p.set_defaults(func=cmd_get_subsets)

    p = sub.add_parser("get-price-guide", help="GET /items/{type}/{no}/price")
    p.add_argument("type")
    p.add_argument("no")
    p.add_argument("--color-id", type=int, default=None)
    p.add_argument("--guide-type", choices=["stock", "sold"], default=None)
    p.add_argument("--new-or-used", choices=["N", "U"], default=None)
    p.add_argument("--country-code", default=None)
    p.add_argument("--region", default=None, help="asia, africa, north_america, south_america, middle_east, europe, eu, oceania")
    p.add_argument("--currency-code", default=None)
    p.add_argument("--vat", choices=["N", "Y", "O"], default=None)
    p.set_defaults(func=cmd_get_price_guide)

    p = sub.add_parser("get-known-colors", help="GET /items/{type}/{no}/colors")
    p.add_argument("type")
    p.add_argument("no")
    p.set_defaults(func=cmd_get_known_colors)

    p = sub.add_parser("get-order-messages", help="GET /orders/{order_id}/messages")
    p.add_argument("order_id", type=int)
    p.set_defaults(func=cmd_get_order_messages)

    p = sub.add_parser("get-order-feedback", help="GET /orders/{order_id}/feedback")
    p.add_argument("order_id", type=int)
    p.set_defaults(func=cmd_get_order_feedback)

    p = sub.add_parser("get-feedback", help="GET /feedback")
    p.add_argument("--direction", choices=["in", "out"], default=None)
    p.set_defaults(func=cmd_get_feedback)

    p = sub.add_parser("get-feedback-item", help="GET /feedback/{feedback_id}")
    p.add_argument("feedback_id", type=int)
    p.set_defaults(func=cmd_get_feedback_item)

    p = sub.add_parser("get-notifications", help="GET /notifications")
    p.set_defaults(func=cmd_get_notifications)

    p = sub.add_parser("get-categories", help="GET /categories")
    p.set_defaults(func=cmd_get_categories)

    p = sub.add_parser("get-colors", help="GET /colors")
    p.set_defaults(func=cmd_get_colors)

    p = sub.add_parser("get-color", help="GET /colors/{color_id}")
    p.add_argument("color_id", type=int)
    p.set_defaults(func=cmd_get_color)

    p = sub.add_parser("get-category", help="GET /categories/{category_id}")
    p.add_argument("category_id", type=int)
    p.set_defaults(func=cmd_get_category)

    p = sub.add_parser("create-inventory", help="POST /inventories (single via flags, or batch via --file)")
    p.add_argument("--file", default=None, help="Batch mode: .json file with array of inventory objects (workspace or /tmp only)")
    _add_inventory_common_args(p, include_item=True)
    p.set_defaults(func=cmd_create_inventory)

    p = sub.add_parser("update-inventory", help="PUT /inventories/{inventory_id}")
    p.add_argument("inventory_id", type=int)
    _add_inventory_common_args(p, include_item=False)
    p.set_defaults(func=cmd_update_inventory)

    p = sub.add_parser("delete-inventory", help="DELETE /inventories/{inventory_id}")
    p.add_argument("inventory_id", type=int)
    p.set_defaults(func=cmd_delete_inventory)

    p = sub.add_parser("update-order", help="PUT /orders/{order_id} (update tracking, remarks, costs, filed)")
    p.add_argument("order_id", type=int)
    p.add_argument("--remarks", default=None)
    p.add_argument("--is-filed", default=None, choices=["true", "false"], help="Set is_filed")
    p.add_argument("--shipping-date-shipped", default=None, help="Timestamp (ISO) for shipping.date_shipped")
    p.add_argument("--shipping-tracking-no", default=None)
    p.add_argument("--shipping-tracking-link", default=None)
    p.add_argument("--shipping-method-id", type=int, default=None)
    p.add_argument("--cost-shipping", default=None)
    p.add_argument("--cost-insurance", default=None)
    p.add_argument("--cost-credit", default=None)
    p.add_argument("--cost-etc1", default=None)
    p.add_argument("--cost-etc2", default=None)
    p.set_defaults(func=cmd_update_order)

    p = sub.add_parser("update-order-status", help="PUT /orders/{order_id}/status")
    p.add_argument("order_id", type=int)
    p.add_argument("status", help="New status value (see BrickLink available status list)")
    p.set_defaults(func=cmd_update_order_status)

    p = sub.add_parser("update-payment-status", help="PUT /orders/{order_id}/payment_status")
    p.add_argument("order_id", type=int)
    p.add_argument("payment_status", help="New payment status value (see BrickLink available status list)")
    p.set_defaults(func=cmd_update_payment_status)

    p = sub.add_parser("send-drive-thru", help="POST /orders/{order_id}/drive_thru")
    p.add_argument("order_id", type=int)
    p.add_argument("--mail-me", action="store_true", help="CC yourself")
    p.set_defaults(func=cmd_send_drive_thru)

    p = sub.add_parser("post-feedback", help="POST /feedback")
    p.add_argument("--order-id", type=int, default=None)
    p.add_argument("--rating", type=int, choices=[0, 1, 2], default=None)
    p.add_argument("--comment", default=None)
    p.set_defaults(func=cmd_post_feedback)

    p = sub.add_parser("reply-feedback", help="POST /feedback/{feedback_id}/reply")
    p.add_argument("feedback_id", type=int)
    p.add_argument("--reply", default=None)
    p.set_defaults(func=cmd_reply_feedback)

    p = sub.add_parser("order-detail-html", help="Fetch order + items and render an HTML list similar to orderDetail.asp")
    p.add_argument("order_id", type=int)
    p.add_argument("--out", default=None, help="Output HTML path (default: /tmp/bricklink_order_<id>_detail.html)")
    p.add_argument("--inline-images", action="store_true", help="Embed images as data: URIs (for sandboxed viewers)")
    p.set_defaults(func=cmd_order_detail_html)

    args = ap.parse_args()

    # normalize boolean arguments
    if hasattr(args, "filed") and isinstance(args.filed, str):
        args.filed = True if args.filed.lower() == "true" else False
    if hasattr(args, "is_filed") and isinstance(args.is_filed, str):
        args.is_filed = True if args.is_filed.lower() == "true" else False
    for k in ("box", "instruction", "break_minifigs", "break_subsets"):
        if hasattr(args, k) and isinstance(getattr(args, k), str):
            setattr(args, k, True if getattr(args, k).lower() == "true" else False)
    for k in ("bulk", "is_retain", "is_stock_room"):
        if hasattr(args, k) and isinstance(getattr(args, k), str):
            setattr(args, k, True if getattr(args, k).lower() == "true" else False)

    try:
        return int(args.func(args) or 0)
    except BrokenPipeError:
        # Allow piping to head/grep without stack traces (also silence Python's final stdout flush)
        try:
            sys.stdout = open(os.devnull, "w")
        except Exception:
            pass
        return 0


if __name__ == "__main__":
    raise SystemExit(main())
