---
name: brave-search-setup
description: Configure Brave Search API and troubleshoot network/proxy issues for web_search functionality. Use when user needs to (1) Set up Brave Search API key, (2) Fix web_search fetch failures, (3) Configure proxy for OpenClaw tools on macOS with Clash/V2Ray/Surge, or (4) Diagnose "fetch failed" errors with web_search/web_fetch tools.
---

# Brave Search Setup & Proxy Configuration

Setup Brave Search API and resolve network connectivity issues for OpenClaw web tools.

## Prerequisites

- Brave Search API key (get from https://brave.com/search/api/)
- OpenClaw CLI installed
- macOS with proxy client (Clash/V2Ray/Surge) if behind GFW

## Quick Setup

### Step 1: Configure API Key

```bash
# Option A: Via config.patch (key will be stored securely)
openclaw gateway config.patch --raw '{"tools":{"web":{"search":{"apiKey":"YOUR_BRAVE_API_KEY","enabled":true,"provider":"brave"}}}}'
```

Or edit `~/.openclaw/openclaw.json` directly:
```json
{
  "tools": {
    "web": {
      "search": {
        "enabled": true,
        "provider": "brave",
        "apiKey": "YOUR_BRAVE_API_KEY"
      }
    }
  }
}
```

### Step 2: Test Without Proxy

```bash
openclaw web.search --query "test" --count 1
```

If works → Done.
If "fetch failed" → Continue to proxy setup.

## Proxy Setup (macOS)

### Step 3: Detect Proxy Port

Common proxy ports by client:
- Clash: 7890 (HTTP), 7891 (SOCKS5), 7897 (mixed-port)
- Surge: 6152, 6153
- V2Ray: 1080, 10808

Detect actual port:
```bash
# Check if Clash is running
ps aux | grep -i clash

# Find mixed-port from Clash config
cat "~/Library/Application Support/io.github.clash-verge-rev.clash-verge-rev/clash-verge.yaml" | grep mixed-port

# Or test common ports
for port in 7890 7891 7897 6152 6153 1080 10808; do
  if nc -z 127.0.0.1 $port 2>/dev/null; then
    echo "Port $port is open"
  fi
done
```

### Step 4: Set System Proxy

**Method A: launchctl (Recommended - survives restart)**
```bash
# Set for current session and future sessions
launchctl setenv HTTPS_PROXY http://127.0.0.1:7897
launchctl setenv HTTP_PROXY http://127.0.0.1:7897
```

**Method B: Shell export (Session only)**
```bash
export HTTPS_PROXY=http://127.0.0.1:7897
export HTTP_PROXY=http://127.0.0.1:7897
```

**Method C: Add to shell profile (Permanent)**
```bash
echo 'export HTTPS_PROXY=http://127.0.0.1:7897' >> ~/.zshrc
echo 'export HTTP_PROXY=http://127.0.0.1:7897' >> ~/.zshrc
source ~/.zshrc
```

### Step 5: Enable Gateway Restart

```bash
openclaw gateway config.patch --raw '{"commands":{"restart":true}}'
```

### Step 6: Restart Gateway with Proxy

```bash
# Restart to pick up proxy env vars
openclaw gateway restart

# Or use SIGUSR1
kill -USR1 $(pgrep -f "openclaw gateway")
```

### Step 7: Verify

```bash
# Test web search
openclaw web.search --query "Brave Search test" --count 1

# Test web fetch
openclaw web.fetch --url "https://api.search.brave.com" --max-chars 100
```

## Troubleshooting

### "fetch failed" but proxy works in browser

Symptom: Browser can access Google, but OpenClaw tools fail.
Cause: Gateway process started before proxy env vars were set.
Solution: Restart Gateway after setting HTTPS_PROXY.

### Permission denied on Gateway restart

Enable restart command:
```bash
openclaw gateway config.patch --raw '{"commands":{"restart":true}}'
```

### API key errors

Verify key is set:
```bash
openclaw gateway config.get | grep -A5 'web.*search'
```

Test directly with curl:
```bash
curl -s "https://api.search.brave.com/res/v1/web/search?q=test&count=1" \
  -H "Accept: application/json" \
  -H "X-Subscription-Token: YOUR_API_KEY"
```

### Mixed-port vs dedicated ports

Clash "mixed-port" (default 7897) handles both HTTP and SOCKS5.
If using dedicated ports:
- HTTP proxy: 7890
- SOCKS5 proxy: 7891 (requires different handling)

## Advanced: Per-Tool Proxy

Not all tools respect HTTPS_PROXY. For tools that don't:

```bash
# Use proxychains-ng
brew install proxychains-ng

# Configure
sudo tee /usr/local/etc/proxychains.conf <<EOF
strict_chain
proxy_dns
[ProxyList]
http 127.0.0.1 7897
EOF

# Run with proxy
proxychains4 openclaw web.search --query "test"
```

## Workflow Summary

1. **Configure API key** → `config.patch` or edit JSON
2. **Test** → If fails, proxy needed
3. **Detect port** → Check Clash/Surge config
4. **Set env vars** → `launchctl setenv` or shell export
5. **Restart Gateway** → `openclaw gateway restart`
6. **Verify** → Run test search

## References

- Brave Search API Docs: https://api.search.brave.com/app/docs
- OpenClaw Config: https://docs.openclaw.ai/config
- Clash Verge: https://github.com/clash-verge-rev/clash-verge-rev
