import { createServer } from "node:http";
import { readFile } from "node:fs/promises";

import { collect as collectCore, formatUptime } from "./collectors/core.js";
import { collect as collectServices } from "./collectors/services.js";
import { collect as collectEmail } from "./collectors/email.js";
import { collect as collectDocker } from "./collectors/docker.js";
import { collect as collectDevServers } from "./collectors/devservers.js";
import { collect as collectSystem } from "./collectors/system.js";
import { collect as collectSkills } from "./collectors/skills.js";

// Allow self-signed certs for Portainer/UniFi
process.env.NODE_TLS_REJECT_UNAUTHORIZED = "0";

// --- Load Config ---
const configPath = new URL("./config.json", import.meta.url);
let config;
try {
  config = JSON.parse(await readFile(configPath, "utf8"));
} catch (e) {
  console.error(`[bot-status] Failed to load config.json: ${e.message}`);
  process.exit(1);
}

const PORT = process.env.PORT || config.port || 3200;
const CACHE_TTL = config.cache?.ttlMs || 10_000;
const startTime = Date.now();

// --- Background Cache ---
// Always serve from cache. Refresh happens in the background on an interval.
// This way the server NEVER blocks on slow shell commands (email checks, gh auth etc.)
let cachedStatus = null;
let refreshing = false;

async function getCronJobs() {
  // Read directly from OpenClaw's cron jobs file — single source of truth
  const ocHome = config.openclawHome || process.env.OPENCLAW_HOME;
  const cronPath = config.cronJobsPath || (ocHome ? `${ocHome}/cron/jobs.json` : null);
  if (!cronPath) return { jobs: [], note: "no cronJobsPath or openclawHome configured" };
  try {
    const raw = JSON.parse(await readFile(cronPath, "utf8"));
    const allJobs = raw.jobs || [];
    
    // Transform to a clean format, only enabled jobs
    const jobs = allJobs
      .filter(j => j.enabled)
      .map(j => {
        let schedule = "unknown";
        if (j.schedule?.kind === "every") {
          const mins = Math.round(j.schedule.everyMs / 60000);
          schedule = mins >= 60 ? `every ${Math.round(mins/60)}h` : `every ${mins}m`;
        } else if (j.schedule?.kind === "cron") {
          schedule = j.schedule.expr;
          if (j.schedule.tz) schedule += ` (${j.schedule.tz})`;
        } else if (j.schedule?.kind === "at") {
          schedule = `at ${new Date(j.schedule.atMs).toISOString()}`;
        }

        return {
          id: j.id,
          name: j.name,
          schedule,
          enabled: j.enabled,
          lastStatus: j.state?.lastStatus || "unknown",
          nextRun: j.state?.nextRunAtMs ? new Date(j.state.nextRunAtMs).toISOString() : null,
          lastRun: j.state?.lastRunAtMs ? new Date(j.state.lastRunAtMs).toISOString() : null,
        };
      });

    return { jobs };
  } catch {
    return { jobs: [], note: "cron jobs file not found" };
  }
}

async function refreshCache() {
  if (refreshing) return;
  refreshing = true;
  try {
    const [botCore, services, communication, containers, devServers, system, crons, skills] =
      await Promise.all([
        collectCore(config),
        collectServices(config),
        collectEmail(config),
        collectDocker(config),
        collectDevServers(config),
        collectSystem(config),
        getCronJobs(),
        collectSkills(config),
      ]);

    cachedStatus = {
      timestamp: new Date().toISOString(),
      naxon: botCore,
      communication,
      crons,
      services,
      containers,
      devServers,
      system,
      skills,
    };
  } catch (e) {
    console.error(`[bot-status] Refresh error: ${e.message}`);
  } finally {
    refreshing = false;
  }
}

// Initial fetch + periodic background refresh
await refreshCache();
setInterval(refreshCache, CACHE_TTL);

// --- Server ---
const server = createServer((req, res) => {
  // CORS
  res.setHeader("Access-Control-Allow-Origin", "*");
  res.setHeader("Access-Control-Allow-Methods", "GET, OPTIONS");
  res.setHeader("Access-Control-Allow-Headers", "Content-Type");

  if (req.method === "OPTIONS") {
    res.writeHead(204);
    res.end();
    return;
  }

  if (req.url === "/status" || req.url === "/") {
    if (cachedStatus) {
      res.writeHead(200, {
        "Content-Type": "application/json",
        "Cache-Control": `public, max-age=${Math.ceil(CACHE_TTL / 1000)}`,
      });
      res.end(JSON.stringify(cachedStatus));
    } else {
      res.writeHead(503, { "Content-Type": "application/json" });
      res.end(JSON.stringify({ error: "starting up, no data yet" }));
    }
  } else if (req.url === "/health") {
    res.writeHead(200, { "Content-Type": "application/json" });
    res.end(JSON.stringify({ status: "ok", uptime: formatUptime(Date.now() - startTime) }));
  } else {
    res.writeHead(404);
    res.end("Not Found");
  }
});

server.listen(PORT, "0.0.0.0", () => {
  console.log(`[bot-status] ${config.name} Status API`);
  console.log(`[bot-status] Listening on http://0.0.0.0:${PORT}`);
  console.log(`[bot-status] Endpoints: /status, /health`);
  console.log(`[bot-status] Cache TTL: ${CACHE_TTL}ms (background refresh)`);
  console.log(`[bot-status] Services: ${(config.services || []).map((s) => s.name).join(", ")}`);
});
