import { readFile } from "node:fs/promises";
import { exec } from "node:child_process";
import { hostname } from "node:os";

function execAsync(command, options = {}) {
  return new Promise((resolve) => {
    exec(command, options, (err, stdout) => {
      if (err) resolve({ ok: false, output: "" });
      else resolve({ ok: true, output: stdout.toString().trim() });
    });
  });
}

export async function collect(config) {
  try {
    // CPU from /proc/loadavg (non-blocking file read)
    const loadavg = await readFile("/proc/loadavg", "utf8");
    const load1m = parseFloat(loadavg.split(" ")[0]);
    const nprocRes = await execAsync("nproc", { timeout: 2000 });
    const nproc = parseInt(nprocRes.output) || 1;
    const cpuUsed = Math.min(100, Math.round((load1m / nproc) * 100));

    // Memory from /proc/meminfo (non-blocking file read)
    const meminfo = await readFile("/proc/meminfo", "utf8");
    const memTotalKb = parseInt(meminfo.match(/MemTotal:\s+(\d+)/)?.[1] || "0");
    const memAvailKb = parseInt(meminfo.match(/MemAvailable:\s+(\d+)/)?.[1] || "0");
    const memTotal = memTotalKb / 1024 / 1024;
    const memUsed = (memTotalKb - memAvailKb) / 1024 / 1024;

    // Disk
    const diskRes = await execAsync("df -BG / | tail -1", { timeout: 2000 });
    const diskParts = diskRes.output.split(/\s+/);
    const diskTotal = parseInt(diskParts[1]) || 0;
    const diskUsed = parseInt(diskParts[2]) || 0;

    return {
      hostname: hostname(),
      ip: config?.hostIp || "127.0.0.1",
      cpu: cpuUsed,
      memoryUsed: parseFloat(memUsed.toFixed(1)),
      memoryTotal: parseFloat(memTotal.toFixed(1)),
      diskUsed,
      diskTotal,
    };
  } catch (e) {
    return { hostname: hostname(), ip: config?.hostIp || "127.0.0.1", error: e.message };
  }
}
