import { readdir, readFile, access } from "node:fs/promises";
import { exec } from "node:child_process";
import { join } from "node:path";

function execAsync(command, options = {}) {
  return new Promise((resolve) => {
    exec(command, options, (err) => {
      resolve({ ok: !err });
    });
  });
}

export async function collect(config) {
  const skillDirs = config.skillDirs || [];

  const skills = [];

  for (const base of skillDirs) {
    let entries;
    try {
      entries = await readdir(base);
    } catch {
      continue;
    }

    const isCustom = base.includes("workspace");

    for (const name of entries.sort()) {
      const skillMd = join(base, name, "SKILL.md");
      try {
        await access(skillMd);
      } catch {
        continue;
      }

      let content;
      try {
        content = await readFile(skillMd, "utf8");
        content = content.slice(0, 2000); // Only need frontmatter
      } catch {
        continue;
      }

      // Extract description
      const descMatch = content.match(/description:\s*["']?(.+?)["']?\s*\n/);
      const description = descMatch ? descMatch[1].replace(/^["']|["']$/g, "").trim() : "";

      // Extract required bins
      const binsMatch = content.match(/"bins":\s*\[([^\]]*)\]/);
      const requiredBins = [];
      if (binsMatch) {
        const raw = binsMatch[1];
        for (const b of raw.split(",")) {
          const bin = b.trim().replace(/"/g, "");
          if (bin) requiredBins.push(bin);
        }
      }

      // Check bins availability (parallel)
      let binsAvailable = true;
      if (requiredBins.length > 0) {
        const checks = await Promise.all(
          requiredBins.map((b) => execAsync(`which ${b}`, { timeout: 2000 }))
        );
        binsAvailable = checks.every((c) => c.ok);
      }

      skills.push({
        name,
        description: description.slice(0, 100),
        requiredBins,
        available: binsAvailable,
        custom: isCustom,
      });
    }
  }

  const available = skills.filter((s) => s.available);

  return {
    total: skills.length,
    available: available.length,
    skills,
  };
}
