import { exec } from "node:child_process";
import { access } from "node:fs/promises";

function execAsync(command, options = {}) {
  return new Promise((resolve) => {
    exec(command, options, (err, stdout) => {
      if (err) resolve({ ok: false, output: "" });
      else resolve({ ok: true, output: stdout.toString().trim() });
    });
  });
}

async function fetchJson(url, options = {}) {
  const controller = new AbortController();
  const timeout = setTimeout(() => controller.abort(), 5000);
  try {
    const res = await fetch(url, {
      ...options,
      signal: controller.signal,
      headers: { ...options.headers },
    });
    clearTimeout(timeout);
    return { ok: res.ok, status: res.status, data: await res.json().catch(() => null) };
  } catch (e) {
    clearTimeout(timeout);
    return { ok: false, error: e.message };
  }
}

async function checkHttp(svc) {
  const url = svc.url + (svc.healthPath || "");
  const options = {};
  if (svc.method) options.method = svc.method;
  if (svc.headers) options.headers = svc.headers;
  if (svc.body) options.body = svc.body;

  const res = await fetchJson(url, options);
  const ok = res.ok || res.status === 200;

  const result = {
    status: ok ? "reachable" : "unreachable",
    url: svc.url.replace(/^https?:\/\//, ""),
  };
  if (svc.label) result.account = svc.label;
  return result;
}

async function checkCommand(svc) {
  const res = await execAsync(`${svc.command} 2>&1`, { timeout: svc.timeout || 5000 });
  const result = { status: res.ok ? "authenticated" : "error" };
  if (svc.label) result.account = svc.label;
  return result;
}

async function checkFileExists(svc) {
  try {
    await access(svc.path);
    const res = await execAsync(`ls ${svc.path} 2>/dev/null`, { timeout: 2000 });
    const ok = res.ok && (res.output.includes("token") || res.output.length > 10);
    const result = { status: ok ? "authenticated" : "not configured" };
    if (svc.label) result.account = svc.label;
    return result;
  } catch {
    const result = { status: "not configured" };
    if (svc.label) result.account = svc.label;
    return result;
  }
}

async function checkService(svc) {
  switch (svc.type) {
    case "http":
      return checkHttp(svc);
    case "command":
      return checkCommand(svc);
    case "file-exists":
      return checkFileExists(svc);
    default:
      return { status: "unknown", error: `Unknown type: ${svc.type}` };
  }
}

export async function collect(config) {
  const services = config.services || [];
  const results = {};

  const checks = await Promise.all(
    services.map(async (svc) => {
      const result = await checkService(svc);
      return { name: svc.name, result };
    })
  );

  for (const { name, result } of checks) {
    results[name] = result;
  }

  // Preserve google services list for backward compat
  if (results.google && !results.google.services) {
    results.google.services = ["calendar", "gmail", "drive", "docs", "sheets", "contacts"];
  }

  const online = Object.values(results).filter(
    (s) => s.status === "reachable" || s.status === "authenticated"
  ).length;

  results._summary = { online, total: Object.keys(results).filter(k => k !== "_summary").length };
  return results;
}
