import { readFile } from "node:fs/promises";
import { exec } from "node:child_process";

function execAsync(command, options = {}) {
  return new Promise((resolve) => {
    exec(command, options, (err, stdout) => {
      if (err) resolve({ ok: false, output: "" });
      else resolve({ ok: true, output: stdout.toString().trim() });
    });
  });
}

export async function collect(config) {
  const accounts = config.email || [];
  const workspace = config.workspace || ".";

  // Read heartbeat state for lastCheck timestamps
  let lastChecks = {};
  try {
    const hb = JSON.parse(
      await readFile(`${workspace}/memory/heartbeat-state.json`, "utf8")
    );
    lastChecks = hb.lastChecks || {};
  } catch {}

  const lastCheckIso = lastChecks.email
    ? new Date(lastChecks.email < 1e12 ? lastChecks.email * 1000 : lastChecks.email).toISOString()
    : null;

  const results = {};

  // Run all email checks in parallel (non-blocking)
  const checks = await Promise.all(
    accounts.map(async (acct) => {
      const res = await execAsync(acct.command, {
        timeout: acct.timeout || 8000,
        env: { ...process.env, ...acct.env },
      });

      let status = "error";
      let unread = 0;

      if (res.ok) {
        status = "connected";
        const count = parseInt(res.output) || 0;
        unread = count > 0 && res.output !== "No results" ? count : 0;
      }

      return {
        name: acct.name,
        data: { status, unread, address: acct.address, lastCheck: lastCheckIso },
      };
    })
  );

  for (const { name, data } of checks) {
    results[name] = data;
  }

  return { email: results };
}
