async function fetchJson(url, options = {}) {
  const controller = new AbortController();
  const timeout = setTimeout(() => controller.abort(), 5000);
  try {
    const res = await fetch(url, {
      ...options,
      signal: controller.signal,
      headers: { ...options.headers },
    });
    clearTimeout(timeout);
    return { ok: res.ok, status: res.status, data: await res.json().catch(() => null) };
  } catch (e) {
    clearTimeout(timeout);
    return { ok: false, error: e.message };
  }
}

export async function collect(config) {
  const docker = config.docker;
  if (!docker) return [];

  try {
    const filter = docker.containerFilter
      ? `?filters=${encodeURIComponent(JSON.stringify({ name: [docker.containerFilter] }))}`
      : "";

    const res = await fetchJson(
      `${docker.url}/api/endpoints/${docker.endpointId}/docker/containers/json${filter}`,
      { headers: { "X-API-Key": docker.token } }
    );

    if (!res.ok || !Array.isArray(res.data)) return [];

    return res.data.map((c) => ({
      name: c.Names[0]?.replace("/", ""),
      status: c.State,
      health: c.Status.includes("healthy")
        ? "healthy"
        : c.Status.includes("unhealthy")
          ? "unhealthy"
          : "none",
      uptime: c.Status,
      ports: c.Ports?.filter((p) => p.PublicPort).map((p) => p.PublicPort) || [],
    }));
  } catch {
    return [];
  }
}
