import { readFile } from "node:fs/promises";

const startTime = Date.now();

export function formatUptime(ms) {
  const s = Math.floor(ms / 1000);
  const h = Math.floor(s / 3600);
  const m = Math.floor((s % 3600) / 60);
  return h > 0 ? `${h}h ${m}m` : `${m}m`;
}

export async function collect(config) {
  let model = config.model || "unknown";
  // Check auth token if openclawHome is configured
  const ocHome = config.openclawHome || process.env.OPENCLAW_HOME;
  if (ocHome) {
    try {
      const authData = JSON.parse(
        await readFile(`${ocHome}/agents/main/agent/auth-profiles.json`, "utf8")
      );
      if (!authData.profiles?.["anthropic:manual"]?.token) {
        model = "no token";
      }
    } catch {}
  }

  // Read heartbeat state (includes vitals + lastChecks)
  let lastHeartbeat = null;
  let contextPercent = null;
  let contextUsed = null;
  let contextMax = null;

  try {
    const hb = JSON.parse(
      await readFile(`${config.workspace}/memory/heartbeat-state.json`, "utf8")
    );

    // Last heartbeat from vitals or check timestamps
    if (hb.vitals?.updatedAt) {
      lastHeartbeat = hb.vitals.updatedAt;
    } else {
      const lastChecks = hb.lastChecks || {};
      const timestamps = Object.values(lastChecks).filter(Boolean);
      lastHeartbeat = timestamps.length > 0
        ? Math.max(...timestamps) * (timestamps[0] < 1e12 ? 1000 : 1)
        : null;
    }

    // Context vitals (written by the bot during heartbeats)
    if (hb.vitals) {
      contextPercent = hb.vitals.contextPercent ?? null;
      contextUsed = hb.vitals.contextUsed ?? null;
      contextMax = hb.vitals.contextMax ?? null;
      // Use model from vitals if fresher
      if (hb.vitals.model) model = hb.vitals.model;
    }
  } catch {}

  // Calculate next heartbeat (~30 min interval)
  const HEARTBEAT_INTERVAL = 30 * 60 * 1000;
  const nextHeartbeat = lastHeartbeat ? lastHeartbeat + HEARTBEAT_INTERVAL : null;

  return {
    status: "online",
    model,
    uptime: formatUptime(Date.now() - startTime),
    uptimeMs: Date.now() - startTime,
    lastHeartbeat: lastHeartbeat ? new Date(lastHeartbeat).toISOString() : null,
    nextHeartbeat: nextHeartbeat ? new Date(nextHeartbeat).toISOString() : null,
    contextPercent,
    contextUsed,
    contextMax,
  };
}
