#!/bin/bash
# Bob P2P Client Setup Script
# Works on: Ubuntu, macOS, Windows (Git Bash/WSL)

set -e

# Get the directory where this script lives
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
CLIENT_SRC="$SKILL_DIR/client"

# Target installation directory
BOB_DIR="$HOME/.bob-p2p"
CLIENT_DIR="$BOB_DIR/client"
CONFIG_FILE="$CLIENT_DIR/config.json"

echo "╔══════════════════════════════════════════╗"
echo "║       Bob P2P Client Setup               ║"
echo "║   Decentralized API Marketplace          ║"
echo "╚══════════════════════════════════════════╝"
echo ""

# Check for Node.js
if ! command -v node &> /dev/null; then
    echo "❌ Node.js is required but not installed."
    echo "   Install Node.js 18+ from: https://nodejs.org/"
    exit 1
fi

NODE_VERSION=$(node -v | cut -d'v' -f2 | cut -d'.' -f1)
if [ "$NODE_VERSION" -lt 18 ]; then
    echo "❌ Node.js 18+ required. Current: $(node -v)"
    exit 1
fi
echo "✓ Node.js $(node -v) detected"

# Check npm
if ! command -v npm &> /dev/null; then
    echo "❌ npm is required but not installed."
    exit 1
fi
echo "✓ npm $(npm -v) detected"

# Create directory structure
echo ""
echo "📁 Creating directories..."
mkdir -p "$BOB_DIR"
mkdir -p "$BOB_DIR/results"

# Check if client source exists in skill
if [ ! -d "$CLIENT_SRC" ]; then
    echo "❌ Client source not found at $CLIENT_SRC"
    echo "   The skill may be corrupted. Try reinstalling."
    exit 1
fi

# Copy or update client
if [ -d "$CLIENT_DIR" ]; then
    echo ""
    echo "📁 Client already installed at $CLIENT_DIR"
    read -p "   Reinstall/update? [y/N] " -n 1 -r
    echo ""
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        rm -rf "$CLIENT_DIR"
    else
        echo "   Keeping existing installation."
    fi
fi

if [ ! -d "$CLIENT_DIR" ]; then
    echo ""
    echo "📥 Installing Bob P2P Client..."
    cp -r "$CLIENT_SRC" "$CLIENT_DIR"
fi

# Install dependencies
echo ""
echo "📦 Installing Node.js dependencies..."
cd "$CLIENT_DIR"
npm install --silent 2>/dev/null || npm install

# Handle config file
if [ ! -f "$CONFIG_FILE" ]; then
    echo ""
    echo "📝 Creating configuration file..."
    
    # Detect OS for proper path handling
    if [[ "$OSTYPE" == "msys" ]] || [[ "$OSTYPE" == "win32" ]] || [[ "$OSTYPE" == "cygwin" ]]; then
        # Windows - convert path
        RESULTS_PATH=$(cygpath -w "$BOB_DIR/results" 2>/dev/null || echo "$BOB_DIR/results")
    else
        RESULTS_PATH="$BOB_DIR/results"
    fi
    
    cat > "$CONFIG_FILE" << CONFIGEOF
{
    "wallet": {
        "address": "YOUR_SOLANA_WALLET_ADDRESS",
        "privateKey": "YOUR_MNEMONIC_OR_PRIVATE_KEY"
    },
    "token": {
        "symbol": "BOB",
        "mint": "F5k1hJjTsMpw8ATJQ1Nba9dpRNSvVFGRaznjiCNUvghH"
    },
    "aggregators": [
        "https://bob-aggregator-uv67ojrpvq-uc.a.run.app"
    ],
    "solana": {
        "network": "mainnet-beta",
        "rpcUrl": "https://api.mainnet-beta.solana.com",
        "confirmations": 3
    },
    "consumer": {
        "enabled": true,
        "timeout": 30000,
        "retryAttempts": 3,
        "results": {
            "outputPath": "$RESULTS_PATH"
        }
    }
}
CONFIGEOF

    echo ""
    echo "╔══════════════════════════════════════════╗"
    echo "║         CONFIGURATION REQUIRED           ║"
    echo "╚══════════════════════════════════════════╝"
    echo ""
    echo "You need to configure your Solana wallet."
    echo ""
    echo "Supported private key formats:"
    echo "  • Mnemonic: \"word1 word2 word3 ...\" (12 or 24 words)"
    echo "  • Array: [123, 45, 67, ...]"
    echo "  • Base58: \"5Kb8kLf4...\""
    echo ""
    
    # Interactive config
    read -p "Configure wallet now? [Y/n] " -n 1 -r
    echo ""
    if [[ ! $REPLY =~ ^[Nn]$ ]]; then
        echo ""
        read -p "Wallet address: " WALLET_ADDRESS
        echo "Private key (will be visible - mnemonic or key):"
        read -p "> " PRIVATE_KEY
        
        if [ -n "$WALLET_ADDRESS" ] && [ -n "$PRIVATE_KEY" ]; then
            cat > "$CONFIG_FILE" << CONFIGEOF
{
    "wallet": {
        "address": "$WALLET_ADDRESS",
        "privateKey": "$PRIVATE_KEY"
    },
    "token": {
        "symbol": "BOB",
        "mint": "F5k1hJjTsMpw8ATJQ1Nba9dpRNSvVFGRaznjiCNUvghH"
    },
    "aggregators": [
        "https://bob-aggregator-uv67ojrpvq-uc.a.run.app"
    ],
    "solana": {
        "network": "mainnet-beta",
        "rpcUrl": "https://api.mainnet-beta.solana.com",
        "confirmations": 3
    },
    "consumer": {
        "enabled": true,
        "timeout": 30000,
        "retryAttempts": 3,
        "results": {
            "outputPath": "$RESULTS_PATH"
        }
    }
}
CONFIGEOF
            echo ""
            echo "✓ Configuration saved!"
        else
            echo ""
            echo "⚠ Wallet not configured. Edit config manually:"
            echo "  $CONFIG_FILE"
        fi
    else
        echo ""
        echo "⚠ Configure your wallet later by editing:"
        echo "  $CONFIG_FILE"
    fi
else
    echo ""
    echo "✓ Configuration file exists"
fi

echo ""
echo "╔══════════════════════════════════════════╗"
echo "║            SETUP COMPLETE!               ║"
echo "╚══════════════════════════════════════════╝"
echo ""
echo "Installation: $CLIENT_DIR"
echo "Config:       $CONFIG_FILE"
echo "Results:      $BOB_DIR/results"
echo ""
echo "Quick start:"
echo "  1. Get \$BOB tokens (if needed)"
echo "  2. Search APIs:  bash scripts/search.sh"
echo "  3. Call an API:  bash scripts/call.sh <api-id> '<json>'"
echo ""
echo "Token: https://pump.fun/coin/F5k1hJjTsMpw8ATJQ1Nba9dpRNSvVFGRaznjiCNUvghH"
echo ""
