#!/bin/bash
# Configure the Bob P2P client
# Works on: Ubuntu, macOS, Windows (Git Bash/WSL)

CLIENT_DIR="$HOME/.bob-p2p/client"
CONFIG_FILE="$CLIENT_DIR/config.json"

echo "╔══════════════════════════════════════════╗"
echo "║      Bob P2P Client Configuration        ║"
echo "╚══════════════════════════════════════════╝"
echo ""

if [ ! -d "$CLIENT_DIR" ]; then
    echo "❌ Bob P2P client not installed."
    SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
    echo "   Run: bash $SCRIPT_DIR/setup.sh"
    exit 1
fi

if [ ! -f "$CONFIG_FILE" ]; then
    echo "❌ Config file not found. Running setup..."
    SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
    bash "$SCRIPT_DIR/setup.sh"
    exit 0
fi

echo "Config file: $CONFIG_FILE"
echo ""

# Show current wallet (masked)
CURRENT_WALLET=$(grep -o '"address"[[:space:]]*:[[:space:]]*"[^"]*"' "$CONFIG_FILE" | head -1 | sed 's/.*"\([^"]*\)"$/\1/')
if [ -n "$CURRENT_WALLET" ] && [ "$CURRENT_WALLET" != "YOUR_SOLANA_WALLET_ADDRESS" ]; then
    MASKED="${CURRENT_WALLET:0:6}...${CURRENT_WALLET: -4}"
    echo "Current wallet: $MASKED"
else
    echo "Current wallet: ⚠ Not configured"
fi

echo ""
echo "Options:"
echo "  1. Update wallet credentials"
echo "  2. View current config"
echo "  3. Open config file location"
echo "  0. Exit"
echo ""

read -p "Choose option [0-3]: " -n 1 -r
echo ""

case $REPLY in
    1)
        echo ""
        read -p "Wallet address: " WALLET_ADDRESS
        echo "Private key (mnemonic or key):"
        read -p "> " PRIVATE_KEY
        
        if [ -n "$WALLET_ADDRESS" ] && [ -n "$PRIVATE_KEY" ]; then
            # Create new config preserving other settings
            RESULTS_PATH="$HOME/.bob-p2p/results"
            cat > "$CONFIG_FILE" << CONFIGEOF
{
    "wallet": {
        "address": "$WALLET_ADDRESS",
        "privateKey": "$PRIVATE_KEY"
    },
    "token": {
        "symbol": "BOB",
        "mint": "F5k1hJjTsMpw8ATJQ1Nba9dpRNSvVFGRaznjiCNUvghH"
    },
    "aggregators": [
        "http://localhost:8080"
    ],
    "solana": {
        "network": "mainnet-beta",
        "rpcUrl": "https://api.mainnet-beta.solana.com",
        "confirmations": 3
    },
    "consumer": {
        "enabled": true,
        "timeout": 30000,
        "retryAttempts": 3,
        "results": {
            "outputPath": "$RESULTS_PATH"
        }
    }
}
CONFIGEOF
            echo ""
            echo "✓ Wallet updated!"
        else
            echo "❌ Invalid input - both address and key required"
        fi
        ;;
    2)
        echo ""
        # Pretty print config (hide private key)
        if command -v python3 &> /dev/null; then
            cat "$CONFIG_FILE" | python3 -c "
import sys, json
d = json.load(sys.stdin)
if 'wallet' in d and 'privateKey' in d['wallet']:
    d['wallet']['privateKey'] = '***HIDDEN***'
print(json.dumps(d, indent=2))
" 2>/dev/null || cat "$CONFIG_FILE"
        else
            echo "(Private key visible in raw output)"
            cat "$CONFIG_FILE"
        fi
        ;;
    3)
        echo ""
        echo "Config location: $CONFIG_FILE"
        echo ""
        # Try to open file manager
        if [[ "$OSTYPE" == "darwin"* ]]; then
            open "$(dirname "$CONFIG_FILE")"
        elif [[ "$OSTYPE" == "msys" ]] || [[ "$OSTYPE" == "win32" ]]; then
            explorer "$(dirname "$CONFIG_FILE")"
        elif command -v xdg-open &> /dev/null; then
            xdg-open "$(dirname "$CONFIG_FILE")"
        else
            echo "Open this folder: $(dirname "$CONFIG_FILE")"
        fi
        ;;
    0|*)
        echo "Done."
        ;;
esac
