#!/bin/bash
# Call a Bob P2P API
# Works on: Ubuntu, macOS, Windows (Git Bash/WSL)
# Usage: bash scripts/call.sh <api-id> '<json-body>'

CLIENT_DIR="$HOME/.bob-p2p/client"
CONFIG_FILE="$CLIENT_DIR/config.json"

# Check installation
if [ ! -d "$CLIENT_DIR" ]; then
    echo "❌ Bob P2P client not installed."
    SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
    echo "   Run: bash $SCRIPT_DIR/setup.sh"
    exit 1
fi

# Show usage
if [ -z "$1" ]; then
    echo "Usage: bash scripts/call.sh <api-id> '<json-body>'"
    echo ""
    echo "Examples:"
    echo "  bash scripts/call.sh echo-api-v1 '{\"message\":\"Hello!\"}'"
    echo "  bash scripts/call.sh runware-text-to-image-v1 '{\"prompt\":\"sunset over mountains\"}'"
    exit 1
fi

API_ID="$1"
BODY="${2:-{}}"

cd "$CLIENT_DIR"

echo "🔍 Looking up API: $API_ID"

# Get aggregator URL from config (cross-platform grep)
AGGREGATOR=$(cat config.json | grep -o '"http[^"]*"' | head -1 | tr -d '"')
if [ -z "$AGGREGATOR" ]; then
    AGGREGATOR="https://bob-aggregator-uv67ojrpvq-uc.a.run.app"
fi

# Fetch API details
API_INFO=$(curl -s "$AGGREGATOR/api/$API_ID" 2>/dev/null)

if echo "$API_INFO" | grep -q "NOT_FOUND"; then
    echo "❌ API not found: $API_ID"
    echo "   Run 'bash scripts/search.sh' to see available APIs"
    exit 1
fi

if echo "$API_INFO" | grep -q "error"; then
    echo "❌ Could not fetch API info. Is the aggregator running?"
    echo "   Aggregator: $AGGREGATOR"
    exit 1
fi

# Parse provider info (cross-platform compatible)
PROVIDER_URL=$(echo "$API_INFO" | grep -o '"endpoint":"[^"]*"' | cut -d'"' -f4)
PROVIDER_WALLET=$(echo "$API_INFO" | grep -o '"address":"[^"]*"' | head -1 | cut -d'"' -f4)
PRICE=$(echo "$API_INFO" | grep -o '"amount":[0-9.]*' | head -1 | cut -d':' -f2)

if [ -z "$PROVIDER_URL" ] || [ -z "$PROVIDER_WALLET" ]; then
    echo "❌ Could not parse provider details"
    echo "   API response: $API_INFO"
    exit 1
fi

echo "   Provider: $PROVIDER_URL"
echo "   Wallet: $PROVIDER_WALLET"
echo "   Price: $PRICE BOB"
echo ""

# Execute the API call
node src/cli/consumer-execute.js "$API_ID" \
    --config config.json \
    --provider "$PROVIDER_URL" \
    --provider-wallet "$PROVIDER_WALLET" \
    --body "$BODY"
