#!/bin/bash
# Check $BOB token balance
# Works on: Ubuntu, macOS, Windows (Git Bash/WSL)

CLIENT_DIR="$HOME/.bob-p2p/client"
CONFIG_FILE="$CLIENT_DIR/config.json"

if [ ! -f "$CONFIG_FILE" ]; then
    echo "❌ Config file not found."
    SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
    echo "   Run: bash $SCRIPT_DIR/setup.sh"
    exit 1
fi

# Extract wallet address from config
WALLET=$(grep -o '"address"[[:space:]]*:[[:space:]]*"[^"]*"' "$CONFIG_FILE" | head -1 | sed 's/.*"\([^"]*\)"$/\1/')
BOB_MINT="F5k1hJjTsMpw8ATJQ1Nba9dpRNSvVFGRaznjiCNUvghH"

if [ -z "$WALLET" ] || [ "$WALLET" = "YOUR_SOLANA_WALLET_ADDRESS" ]; then
    echo "❌ Wallet not configured."
    echo "   Edit: $CONFIG_FILE"
    exit 1
fi

echo "🔍 Checking balance for: $WALLET"
echo ""

# Query Solana RPC for token balance
RESPONSE=$(curl -s "https://api.mainnet-beta.solana.com" \
    -X POST \
    -H "Content-Type: application/json" \
    -d "{
        \"jsonrpc\": \"2.0\",
        \"id\": 1,
        \"method\": \"getTokenAccountsByOwner\",
        \"params\": [
            \"$WALLET\",
            {\"mint\": \"$BOB_MINT\"},
            {\"encoding\": \"jsonParsed\"}
        ]
    }" 2>/dev/null)

# Parse balance (cross-platform)
BALANCE=$(echo "$RESPONSE" | grep -o '"uiAmountString":"[^"]*"' | head -1 | cut -d'"' -f4)

if [ -z "$BALANCE" ]; then
    echo "💰 \$BOB Balance: 0"
    echo ""
    echo "Get \$BOB tokens at:"
    echo "https://pump.fun/coin/$BOB_MINT"
else
    echo "💰 \$BOB Balance: $BALANCE"
fi

echo ""

# Check SOL balance for gas
SOL_RESPONSE=$(curl -s "https://api.mainnet-beta.solana.com" \
    -X POST \
    -H "Content-Type: application/json" \
    -d "{
        \"jsonrpc\": \"2.0\",
        \"id\": 1,
        \"method\": \"getBalance\",
        \"params\": [\"$WALLET\"]
    }" 2>/dev/null)

SOL_LAMPORTS=$(echo "$SOL_RESPONSE" | grep -o '"value":[0-9]*' | cut -d':' -f2)
if [ -n "$SOL_LAMPORTS" ] && [ "$SOL_LAMPORTS" != "0" ]; then
    # Cross-platform math (awk instead of bc)
    SOL_BALANCE=$(awk "BEGIN {printf \"%.6f\", $SOL_LAMPORTS / 1000000000}")
    echo "⛽ SOL Balance: $SOL_BALANCE SOL (for tx fees)"
else
    echo "⛽ SOL Balance: 0 SOL"
    echo "   ⚠ You need some SOL for transaction fees!"
fi
