# Providing APIs on Bob P2P

This guide covers how to offer your own APIs on the Bob P2P network and earn $BOB tokens.

## Overview

As a provider, you can:
- Offer any API endpoint (AI models, data services, utilities)
- Set your own pricing in $BOB
- Control capacity and queue limits
- Earn tokens automatically when consumers call your APIs

## Prerequisites

1. Bob P2P client installed (`bash scripts/setup.sh`)
2. Wallet with some SOL for transaction fees
3. An API or service to offer

## Configuration

Update your `~/.bob-p2p/client/config.json` to enable provider mode:

```json
{
    "provider": {
        "enabled": true,
        "port": 8000,
        "host": "0.0.0.0",
        "publicEndpoint": "https://your-domain.com:8000",
        "database": {
            "path": "~/.bob-p2p/provider.db"
        },
        "queue": {
            "codeExpiry": 60,
            "maxConcurrent": 2,
            "maxQueueLength": 10
        },
        "results": {
            "retention": 86400,
            "storagePath": "~/.bob-p2p/results",
            "maxStorageGB": 10
        }
    }
}
```

## Defining APIs

Create `~/.bob-p2p/client/api.json` with your API definitions:

```json
{
    "apis": [
        {
            "id": "my-image-gen-v1",
            "name": "My Image Generator",
            "description": "Generate images from text prompts",
            "version": "1.0.0",
            "endpoint": "/generate",
            "method": "POST",
            "handler": "./handlers/my-handler.js",
            "pricing": {
                "amount": 0.1,
                "unit": "per-call"
            },
            "capacity": {
                "concurrent": 2,
                "queueMax": 10,
                "queueTimeout": 60
            },
            "execution": {
                "estimatedDuration": 30,
                "maxDuration": 120,
                "resultRetention": 86400
            },
            "schema": {
                "request": {
                    "type": "object",
                    "properties": {
                        "prompt": { "type": "string", "maxLength": 500 }
                    },
                    "required": ["prompt"]
                },
                "response": {
                    "type": "object",
                    "properties": {
                        "imageUrl": { "type": "string" }
                    }
                }
            },
            "category": ["ml", "image"],
            "tags": ["image-generation", "ai"]
        }
    ]
}
```

## Creating Handlers

Handlers are Node.js modules that implement your API logic:

```javascript
// handlers/my-handler.js
module.exports = async function handler(params, context) {
    const { updateProgress, saveResult, jobId } = context;

    // Update progress
    await updateProgress(10, 'Starting...');

    // Your logic here
    const result = await generateImage(params.prompt);

    // Save result file if needed
    const url = await saveResult(result.buffer, 'output.png');

    await updateProgress(100, 'Complete');

    return { imageUrl: url };
};
```

## Starting Provider

```bash
cd ~/.bob-p2p/client
npm run provide -- --config config.json --apis api.json
```

This will:
1. Start the provider server
2. Register APIs with configured aggregators
3. Begin accepting and processing requests

## Monitoring

### View Earnings
```bash
npm run earnings -- --config config.json
```

### View Queue
```bash
npm run queue-status -- --config config.json
```

### View Jobs
```bash
npm run jobs -- --config config.json
npm run jobs -- --config config.json --status completed
```

## Pricing Guidelines

Consider:
- Compute costs (GPU time, API calls)
- Your desired margin
- Market rates for similar services

Example pricing:
- Simple utility: 0.01-0.05 BOB
- Image generation: 0.05-0.2 BOB
- Video generation: 0.2-1.0 BOB
- Complex ML tasks: 0.5-5.0 BOB

## Public Endpoint

Your `publicEndpoint` must be reachable by consumers. Options:
- Public server with domain
- ngrok for development: `ngrok http 8000`
- Cloudflare tunnel

## Security

- Validate all inputs in your handlers
- Set appropriate rate limits
- Monitor for abuse
- Keep your private key secure

## Registering with Aggregators

Your APIs are automatically registered with aggregators in your config when you start the provider. To add more aggregators:

```json
{
    "aggregators": [
        "http://localhost:8080",
        "https://aggregator.bob-p2p.network"
    ]
}
```
