/**
 * Configuration Loader
 */

const fs = require('fs');
const path = require('path');

function loadConfig(configPath) {
    if (!configPath) {
        throw new Error('Config file path is required. Use --config <path>');
    }

    if (!fs.existsSync(configPath)) {
        throw new Error(`Config file not found: ${configPath}`);
    }

    try {
        const content = fs.readFileSync(configPath, 'utf8');
        const config = JSON.parse(content);

        // Validate required fields
        validateConfig(config);

        return config;

    } catch (error) {
        if (error instanceof SyntaxError) {
            throw new Error(`Invalid JSON in config file: ${error.message}`);
        }
        throw error;
    }
}

function validateConfig(config) {
    // Wallet
    if (!config.wallet) {
        throw new Error('Config missing: wallet');
    }
    if (!config.wallet.address) {
        throw new Error('Config missing: wallet.address');
    }
    if (!config.wallet.privateKey) {
        throw new Error('Config missing: wallet.privateKey');
    }

    // Token
    if (!config.token) {
        throw new Error('Config missing: token');
    }
    if (!config.token.symbol) {
        throw new Error('Config missing: token.symbol');
    }
    if (!config.token.mint) {
        throw new Error('Config missing: token.mint');
    }

    // Aggregators
    if (!config.aggregators || !Array.isArray(config.aggregators)) {
        throw new Error('Config missing or invalid: aggregators (must be array)');
    }

    // Solana
    if (!config.solana) {
        throw new Error('Config missing: solana');
    }
    if (!config.solana.network) {
        throw new Error('Config missing: solana.network');
    }
    if (!config.solana.rpcUrl) {
        throw new Error('Config missing: solana.rpcUrl');
    }

    // Provider (if enabled)
    if (config.provider && config.provider.enabled) {
        // HTTP port required unless HTTP is disabled
        if (!config.provider.httpDisabled && !config.provider.port) {
            throw new Error('Config missing: provider.port (or set httpDisabled: true)');
        }
        // publicEndpoint required unless HTTP is disabled or P2P is enabled
        const p2pEnabled = config.p2p && config.p2p.enabled !== false;
        if (!config.provider.httpDisabled && !p2pEnabled && !config.provider.publicEndpoint) {
            throw new Error('Config missing: provider.publicEndpoint (or enable P2P or set httpDisabled: true)');
        }
        if (!config.provider.database) {
            throw new Error('Config missing: provider.database');
        }
        if (!config.provider.database.type) {
            throw new Error('Config missing: provider.database.type');
        }
        if (!config.provider.results) {
            throw new Error('Config missing: provider.results');
        }
        if (!config.provider.results.storagePath) {
            throw new Error('Config missing: provider.results.storagePath');
        }
    }

    // P2P (if enabled)
    if (config.p2p && config.p2p.enabled !== false) {
        // P2P config is mostly optional with sensible defaults
        // Validate bootstrap peers if provided
        if (config.p2p.bootstrap && !Array.isArray(config.p2p.bootstrap)) {
            throw new Error('Config invalid: p2p.bootstrap must be an array');
        }
    }

    // Consumer (if enabled)
    if (config.consumer && config.consumer.enabled) {
        // Consumer config is mostly optional with defaults
    }

    console.log('Configuration validated successfully');
}

function loadApiDefinitions(apiPath) {
    if (!apiPath) {
        throw new Error('API definitions file path is required. Use --apis <path>');
    }

    if (!fs.existsSync(apiPath)) {
        throw new Error(`API definitions file not found: ${apiPath}`);
    }

    try {
        const content = fs.readFileSync(apiPath, 'utf8');
        const apiDef = JSON.parse(content);

        if (!apiDef.apis || !Array.isArray(apiDef.apis)) {
            throw new Error('API definitions must contain "apis" array');
        }

        // Validate each API
        for (const api of apiDef.apis) {
            validateApiDefinition(api);
        }

        return apiDef;

    } catch (error) {
        if (error instanceof SyntaxError) {
            throw new Error(`Invalid JSON in API definitions file: ${error.message}`);
        }
        throw error;
    }
}

function validateApiDefinition(api) {
    const required = [
        'id',
        'name',
        'description',
        'version',
        'endpoint',
        'method',
        'handler',
        'pricing',
        'capacity',
        'execution',
        'schema'
    ];

    for (const field of required) {
        if (!api[field]) {
            throw new Error(`API ${api.id || 'unknown'} missing required field: ${field}`);
        }
    }

    // Validate pricing
    if (typeof api.pricing.amount !== 'number' || api.pricing.amount < 0) {
        throw new Error(`API ${api.id} has invalid pricing.amount`);
    }

    // Validate capacity
    if (typeof api.capacity.concurrent !== 'number' || api.capacity.concurrent < 1) {
        throw new Error(`API ${api.id} has invalid capacity.concurrent`);
    }
    if (typeof api.capacity.queueMax !== 'number' || api.capacity.queueMax < 0) {
        throw new Error(`API ${api.id} has invalid capacity.queueMax`);
    }

    // Validate execution
    if (typeof api.execution.maxDuration !== 'number' || api.execution.maxDuration < 1) {
        throw new Error(`API ${api.id} has invalid execution.maxDuration`);
    }

    // Validate schema
    if (!api.schema.request || !api.schema.response) {
        throw new Error(`API ${api.id} missing request or response schema`);
    }

    // Validate handler exists
    if (!fs.existsSync(api.handler)) {
        throw new Error(`API ${api.id} handler not found: ${api.handler}`);
    }
}

module.exports = {
    loadConfig,
    loadApiDefinitions
};
