/**
 * Solana Integration Module
 *
 * Handles payment verification and transactions
 */

const {
    Connection,
    PublicKey,
    Keypair,
    Transaction,
    SystemProgram,
    LAMPORTS_PER_SOL
} = require('@solana/web3.js');

const {
    getAssociatedTokenAddress,
    createTransferInstruction,
    TOKEN_PROGRAM_ID
} = require('@solana/spl-token');

class SolanaClient {
    constructor(config) {
        this.config = config;
        this.connection = new Connection(
            config.solana.rpcUrl,
            'confirmed'
        );

        // Load wallet
        this.keypair = this.loadKeypair(config.wallet.privateKey);

        this.publicKey = this.keypair.publicKey;
        this.tokenMint = new PublicKey(config.token.mint);

        console.log(`Solana client initialized`);
        console.log(`Wallet: ${this.publicKey.toBase58()}`);
        console.log(`Network: ${config.solana.network}`);
    }

    /**
     * Load keypair from various formats
     *
     * @param {*} privateKey - Private key in various formats
     * @returns {Keypair} - Solana keypair
     */
    loadKeypair(privateKey) {
        // Array format: [123, 45, 67, ...]
        if (Array.isArray(privateKey)) {
            return Keypair.fromSecretKey(Uint8Array.from(privateKey));
        }

        // String format - need to detect type
        if (typeof privateKey === 'string') {
            const trimmed = privateKey.trim();

            // Check if it's a mnemonic phrase (12 or 24 words)
            const words = trimmed.split(/\s+/);
            if (words.length === 12 || words.length === 24) {
                return this.keypairFromMnemonic(trimmed);
            }

            // Base58 format (starts with numbers/letters, no spaces)
            const bs58 = require('bs58');
            try {
                return Keypair.fromSecretKey(bs58.decode(trimmed));
            } catch (error) {
                throw new Error('Invalid private key format. Must be: array [123, 45, ...], base58 string, or mnemonic phrase');
            }
        }

        throw new Error('Invalid private key format. Must be: array [123, 45, ...], base58 string, or mnemonic phrase');
    }

    /**
     * Create keypair from mnemonic phrase
     *
     * @param {string} mnemonic - BIP39 mnemonic phrase
     * @returns {Keypair} - Solana keypair
     */
    keypairFromMnemonic(mnemonic) {
        const bip39 = require('bip39');
        const { derivePath } = require('ed25519-hd-key');

        // Validate mnemonic
        if (!bip39.validateMnemonic(mnemonic)) {
            throw new Error('Invalid mnemonic phrase');
        }

        // Derive seed from mnemonic
        const seed = bip39.mnemonicToSeedSync(mnemonic, '');

        // Use standard Solana derivation path
        const path = "m/44'/501'/0'/0'";
        const derivedSeed = derivePath(path, seed.toString('hex')).key;

        // Create and return keypair
        return Keypair.fromSeed(derivedSeed);
    }

    /**
     * Verify a payment transaction
     *
     * @param {string} signature - Transaction signature
     * @param {string} expectedRecipient - Expected recipient address
     * @param {number} expectedAmount - Expected amount in tokens
     * @returns {Promise<boolean>} - True if valid payment
     */
    async verifyPayment(signature, expectedRecipient, expectedAmount) {
        try {
            // Get transaction details
            const tx = await this.connection.getTransaction(signature, {
                commitment: 'confirmed',
                maxSupportedTransactionVersion: 0
            });

            if (!tx) {
                console.log(`Transaction not found: ${signature}`);
                return false;
            }

            // Check confirmation status
            const currentSlot = await this.connection.getSlot();
            const requiredConfirmations = this.config.solana.confirmations || 1;

            if (currentSlot - tx.slot < requiredConfirmations) {
                console.log(`Transaction not confirmed yet (slot: ${tx.slot}, current: ${currentSlot})`);
                return false;
            }

            // Check transaction success
            if (tx.meta.err) {
                console.log(`Transaction failed: ${JSON.stringify(tx.meta.err)}`);
                return false;
            }

            // Parse transfer instruction
            const transfer = this.parseTransferInstruction(tx);

            if (!transfer) {
                console.log(`No valid transfer instruction found`);
                return false;
            }

            // Verify recipient
            if (transfer.recipient !== expectedRecipient) {
                console.log(`Recipient mismatch: ${transfer.recipient} != ${expectedRecipient}`);
                return false;
            }

            // Verify token mint
            if (transfer.tokenMint !== this.tokenMint.toBase58()) {
                console.log(`Token mint mismatch: ${transfer.tokenMint} != ${this.tokenMint.toBase58()}`);
                return false;
            }

            // Verify amount (allow overpayment)
            if (transfer.amount < expectedAmount) {
                console.log(`Amount too low: ${transfer.amount} < ${expectedAmount}`);
                return false;
            }

            console.log(`Payment verified: ${transfer.amount} tokens to ${transfer.recipient}`);
            return true;

        } catch (error) {
            console.error(`Payment verification error:`, error);
            return false;
        }
    }

    /**
     * Parse transfer instruction from transaction
     */
    parseTransferInstruction(tx) {
        try {
            // Look for SPL token transfer instruction
            const instructions = tx.transaction.message.instructions;
            const accountKeys = tx.transaction.message.accountKeys;

            for (const instruction of instructions) {
                const programId = accountKeys[instruction.programIdIndex];

                // Check if this is a token program instruction
                if (programId.toBase58() === TOKEN_PROGRAM_ID.toBase58()) {
                    // Parse the instruction data - decode from base58 if it's a string
                    let data = instruction.data;
                    if (typeof data === 'string') {
                        const bs58 = require('bs58');
                        data = bs58.decode(data);
                    }

                    // Transfer instruction type is 3
                    if (data[0] === 3) {
                        // Amount is bytes 1-8 (little endian u64)
                        const amountBuffer = data.slice(1, 9);
                        const amount = Number(
                            amountBuffer[0] +
                            amountBuffer[1] * 256 +
                            amountBuffer[2] * 256 * 256 +
                            amountBuffer[3] * 256 * 256 * 256
                        );

                        // Get source and destination accounts
                        const sourceIndex = instruction.accounts[0];
                        const destIndex = instruction.accounts[1];

                        // Get token mint and recipient owner from postTokenBalances
                        let tokenMint = null;
                        let recipient = null;

                        if (tx.meta.postTokenBalances) {
                            // Find the destination token account in postTokenBalances
                            for (const balance of tx.meta.postTokenBalances) {
                                if (balance.accountIndex === destIndex) {
                                    tokenMint = balance.mint;
                                    recipient = balance.owner;  // This is the wallet owner, not the token account
                                    break;
                                }
                            }
                        }

                        if (!recipient) {
                            console.log('Could not find recipient owner in token balances');
                            return null;
                        }

                        return {
                            amount: amount / Math.pow(10, 6), // Convert from smallest unit (BOB has 6 decimals)
                            recipient,
                            tokenMint,
                            source: accountKeys[sourceIndex].toBase58()
                        };
                    }
                }
            }

            return null;

        } catch (error) {
            console.error('Error parsing transfer instruction:', error);
            return null;
        }
    }

    /**
     * Send payment to a provider
     *
     * @param {string} recipient - Recipient address
     * @param {number} amount - Amount in tokens
     * @returns {Promise<string>} - Transaction signature
     */
    async sendPayment(recipient, amount) {
        try {
            const recipientPubkey = new PublicKey(recipient);

            // Get associated token accounts
            const sourceTokenAccount = await getAssociatedTokenAddress(
                this.tokenMint,
                this.publicKey
            );

            const destTokenAccount = await getAssociatedTokenAddress(
                this.tokenMint,
                recipientPubkey
            );

            // Create transfer instruction
            const transferInstruction = createTransferInstruction(
                sourceTokenAccount,
                destTokenAccount,
                this.publicKey,
                amount * Math.pow(10, 9) // Convert to smallest unit
            );

            // Create and send transaction
            const transaction = new Transaction().add(transferInstruction);

            const signature = await this.connection.sendTransaction(
                transaction,
                [this.keypair],
                { skipPreflight: false }
            );

            console.log(`Payment sent: ${signature}`);

            // Wait for confirmation
            await this.connection.confirmTransaction(signature, 'confirmed');

            console.log(`Payment confirmed: ${signature}`);

            return signature;

        } catch (error) {
            console.error('Payment error:', error);
            throw new Error(`Payment failed: ${error.message}`);
        }
    }

    /**
     * Get token balance
     */
    async getBalance() {
        try {
            const tokenAccount = await getAssociatedTokenAddress(
                this.tokenMint,
                this.publicKey
            );

            const balance = await this.connection.getTokenAccountBalance(tokenAccount);
            return parseFloat(balance.value.uiAmount);

        } catch (error) {
            console.error('Error getting balance:', error);
            return 0;
        }
    }

    /**
     * Get SOL balance
     */
    async getSolBalance() {
        try {
            const balance = await this.connection.getBalance(this.publicKey);
            return balance / LAMPORTS_PER_SOL;
        } catch (error) {
            console.error('Error getting SOL balance:', error);
            return 0;
        }
    }
}

module.exports = SolanaClient;
