/**
 * Queue Management System
 *
 * Handles queue codes, position tracking, and capacity management
 */

const { generateQueueCode, calculateExpiry, isExpired } = require('../utils/helpers');

class QueueManager {
    constructor(database, config) {
        this.database = database;
        this.config = config;
        this.apis = new Map(); // apiId -> API definition

        // Start cleanup interval
        this.startCleanup();
    }

    /**
     * Register an API
     */
    registerApi(api) {
        this.apis.set(api.id, api);
        console.log(`API registered: ${api.id} (${api.name})`);
    }

    /**
     * Get API definition
     */
    getApi(apiId) {
        return this.apis.get(apiId);
    }

    /**
     * Get all APIs
     */
    getAllApis() {
        return Array.from(this.apis.values());
    }

    /**
     * Request a queue position
     *
     * @param {string} apiId - API identifier
     * @param {string} walletAddress - Consumer wallet address
     * @returns {Promise<object>} - Queue code, position, price, expiry
     */
    async requestQueue(apiId, walletAddress) {
        const api = this.apis.get(apiId);

        if (!api) {
            throw new Error(`API not found: ${apiId}`);
        }

        // Get current queue position
        const currentPosition = await this.database.getQueuePosition(apiId);

        // Check if queue is full
        if (currentPosition >= api.capacity.queueMax) {
            throw new Error(`Queue full for ${apiId} (max: ${api.capacity.queueMax})`);
        }

        // Generate queue code
        const code = generateQueueCode();
        const position = currentPosition + 1;
        const price = api.pricing.amount;
        const expirySeconds = api.capacity.queueTimeout || 60;
        const expiresAt = calculateExpiry(expirySeconds);

        // Store queue code
        await this.database.createQueueCode({
            code,
            apiId,
            walletAddress,
            price,
            position,
            createdAt: Date.now(),
            expiresAt
        });

        return {
            code,
            position,
            price,
            expiresAt,
            expirySeconds
        };
    }

    /**
     * Validate queue code
     *
     * @param {string} code - Queue code
     * @param {string} walletAddress - Wallet address to verify
     * @returns {Promise<object|null>} - Queue code data or null if invalid
     */
    async validateQueueCode(code, walletAddress) {
        const queueCode = await this.database.getQueueCode(code);

        if (!queueCode) {
            return null;
        }

        // Check expiry
        if (isExpired(queueCode.expiresAt)) {
            await this.database.deleteQueueCode(code);
            return null;
        }

        // Check wallet address
        if (queueCode.walletAddress !== walletAddress) {
            return null;
        }

        return queueCode;
    }

    /**
     * Consume queue code (after successful payment verification)
     *
     * @param {string} code - Queue code to consume
     */
    async consumeQueueCode(code) {
        await this.database.deleteQueueCode(code);
    }

    /**
     * Get API status
     *
     * @param {string} apiId - API identifier
     * @returns {Promise<object>} - API status
     */
    async getApiStatus(apiId) {
        const api = this.apis.get(apiId);

        if (!api) {
            throw new Error(`API not found: ${apiId}`);
        }

        // Get queue length
        const queueLength = await this.database.getQueuePosition(apiId);

        // Get active jobs
        const activeJobs = await this.database.getJobsByStatus('processing');
        const apiActiveJobs = activeJobs.filter(job => job.apiId === apiId).length;

        // Calculate availability
        const available = apiActiveJobs < api.capacity.concurrent;
        const queueAvailable = queueLength < api.capacity.queueMax;

        return {
            apiId: api.id,
            name: api.name,
            description: api.description,
            version: api.version,
            price: api.pricing.amount,
            unit: api.pricing.unit,
            capacity: {
                concurrent: api.capacity.concurrent,
                queueMax: api.capacity.queueMax,
                queueTimeout: api.capacity.queueTimeout
            },
            currentQueue: queueLength,
            activeJobs: apiActiveJobs,
            available,
            queueAvailable,
            estimatedWait: this.estimateWaitTime(api, queueLength, apiActiveJobs)
        };
    }

    /**
     * Estimate wait time for queue
     */
    estimateWaitTime(api, queueLength, activeJobs) {
        if (activeJobs < api.capacity.concurrent && queueLength === 0) {
            return 0; // Immediate
        }

        const avgDuration = api.execution.estimatedDuration || 60;
        const positionsAhead = queueLength + (activeJobs - api.capacity.concurrent);

        return Math.max(0, positionsAhead * avgDuration);
    }

    /**
     * Cleanup expired queue codes
     */
    async cleanup() {
        const deleted = await this.database.cleanupExpiredQueueCodes();
        if (deleted > 0) {
            console.log(`Cleaned up ${deleted} expired queue codes`);
        }
    }

    /**
     * Start automatic cleanup
     */
    startCleanup() {
        setInterval(() => {
            this.cleanup().catch(err => {
                console.error('Queue cleanup error:', err);
            });
        }, 10000); // Every 10 seconds
    }
}

module.exports = QueueManager;
