/**
 * Provider Server
 *
 * Exposes API endpoints for consumers to interact with
 */

const express = require('express');
const path = require('path');
const fs = require('fs');

class ProviderServer {
    constructor(config, queueManager, paymentVerifier, jobExecutor) {
        this.config = config;
        this.queue = queueManager;
        this.payment = paymentVerifier;
        this.jobs = jobExecutor;

        this.app = express();
        this.app.use(express.json({ limit: '10mb' }));

        this.setupRoutes();
    }

    setupRoutes() {
        // Health check
        this.app.get('/health', (req, res) => {
            res.json({
                status: 'ok',
                uptime: process.uptime(),
                timestamp: Date.now()
            });
        });

        // Provider info
        this.app.get('/info', (req, res) => {
            const apis = this.queue.getAllApis().map(api => ({
                id: api.id,
                name: api.name,
                description: api.description,
                version: api.version,
                endpoint: `/api/${api.id}`,
                price: api.pricing.amount,
                unit: api.pricing.unit,
                category: api.category,
                tags: api.tags
            }));

            res.json({
                provider: this.config.wallet.address,
                endpoint: this.config.provider.publicEndpoint,
                apis,
                token: {
                    symbol: this.config.token.symbol,
                    mint: this.config.token.mint
                }
            });
        });

        // API status
        this.app.get('/api/:apiId/status', async (req, res) => {
            try {
                const { apiId } = req.params;
                const status = await this.queue.getApiStatus(apiId);

                res.json(status);

            } catch (error) {
                res.status(404).json({
                    error: error.message
                });
            }
        });

        // Request queue position
        this.app.post('/api/:apiId/queue', async (req, res) => {
            try {
                const { apiId } = req.params;
                const { walletAddress } = req.body;

                if (!walletAddress) {
                    return res.status(400).json({
                        error: 'Missing walletAddress'
                    });
                }

                const queueData = await this.queue.requestQueue(apiId, walletAddress);

                res.json(queueData);

            } catch (error) {
                res.status(400).json({
                    error: error.message
                });
            }
        });

        // Execute API
        this.app.post('/api/:apiId/execute', async (req, res) => {
            try {
                const { apiId } = req.params;
                const {
                    queueCode,
                    transactionSignature,
                    walletAddress,
                    params
                } = req.body;

                // Validate required fields
                if (!queueCode || !transactionSignature || !walletAddress || !params) {
                    return res.status(400).json({
                        error: 'Missing required fields: queueCode, transactionSignature, walletAddress, params'
                    });
                }

                // Validate queue code
                const queueData = await this.queue.validateQueueCode(queueCode, walletAddress);

                if (!queueData) {
                    return res.status(400).json({
                        error: 'Invalid or expired queue code'
                    });
                }

                // Verify queue code is for correct API
                if (queueData.apiId !== apiId) {
                    return res.status(400).json({
                        error: 'Queue code is for different API'
                    });
                }

                // Verify payment
                const paymentValid = await this.payment.verifyPayment(
                    transactionSignature,
                    queueData.price,
                    queueData
                );

                if (!paymentValid) {
                    return res.status(402).json({
                        error: 'Payment verification failed'
                    });
                }

                // Create job
                const jobId = await this.jobs.createJob(
                    apiId,
                    params,
                    walletAddress,
                    transactionSignature
                );

                // Mark transaction as used
                await this.payment.markTransactionUsed(
                    transactionSignature,
                    jobId,
                    queueData.price
                );

                // Record earning
                await this.payment.recordEarning(
                    apiId,
                    jobId,
                    queueData.price,
                    walletAddress,
                    transactionSignature
                );

                // Consume queue code
                await this.queue.consumeQueueCode(queueCode);

                // Return job ID
                res.json({
                    jobId,
                    status: 'queued',
                    message: 'Job created successfully'
                });

            } catch (error) {
                console.error('Execute API error:', error);
                res.status(500).json({
                    error: error.message
                });
            }
        });

        // Get job status
        this.app.get('/job/:jobId', async (req, res) => {
            try {
                const { jobId } = req.params;
                const job = await this.jobs.getJob(jobId);

                res.json({
                    jobId: job.jobId,
                    apiId: job.apiId,
                    status: job.status,
                    progress: job.progress,
                    progressMessage: job.progressMessage,
                    result: job.result,
                    resultFilename: job.resultFilename,
                    error: job.error,
                    createdAt: job.createdAt,
                    startedAt: job.startedAt,
                    completedAt: job.completedAt
                });

            } catch (error) {
                res.status(404).json({
                    error: error.message
                });
            }
        });

        // Download job result file (P2P streaming)
        this.app.get('/job/:jobId/download', async (req, res) => {
            try {
                const { jobId } = req.params;
                const job = await this.jobs.getJob(jobId);

                if (job.status !== 'completed') {
                    return res.status(400).json({
                        error: 'Job not completed yet'
                    });
                }

                if (!job.resultFilename) {
                    return res.status(404).json({
                        error: 'No result file available for this job'
                    });
                }

                const filepath = path.join(this.jobs.resultStorage, job.resultFilename);

                if (!fs.existsSync(filepath)) {
                    return res.status(404).json({
                        error: 'Result file not found on disk'
                    });
                }

                // Stream the file to consumer
                res.sendFile(filepath);

            } catch (error) {
                res.status(404).json({
                    error: error.message
                });
            }
        });

        // Serve result files
        this.app.get('/results/:filename', (req, res) => {
            const { filename } = req.params;
            const filepath = path.join(this.jobs.resultStorage, filename);

            if (!fs.existsSync(filepath)) {
                return res.status(404).json({
                    error: 'Result file not found'
                });
            }

            res.sendFile(filepath);
        });

        // 404 handler
        this.app.use((req, res) => {
            res.status(404).json({
                error: 'Endpoint not found'
            });
        });

        // Error handler
        this.app.use((err, req, res, next) => {
            console.error('Server error:', err);
            res.status(500).json({
                error: 'Internal server error'
            });
        });
    }

    start() {
        const port = this.config.provider.port;

        this.app.listen(port, '0.0.0.0', () => {
            console.log(`Provider server listening on http://0.0.0.0:${port}`);
            console.log(`Public endpoint: ${this.config.provider.publicEndpoint}`);
        });
    }
}

module.exports = ProviderServer;
