/**
 * P2P Provider Server
 *
 * Handles API requests over P2P network
 */

const P2PNode = require('../p2p/node');
const {
    BOB_API_PROTOCOL,
    createApiRequestHandler,
    createResultFetchHandler
} = require('../p2p/protocols');

class P2PProviderServer {
    constructor(config, queueManager, paymentVerifier, jobExecutor) {
        this.config = config;
        this.queue = queueManager;
        this.payment = paymentVerifier;
        this.jobs = jobExecutor;
        this.p2pNode = null;
    }

    /**
     * Start the P2P server
     */
    async start() {
        console.log('Starting P2P Provider Server...');

        // Create and start P2P node
        this.p2pNode = new P2PNode(this.config);
        await this.p2pNode.start();

        // Register API request handler
        const apiRequestHandler = createApiRequestHandler(this.queue);
        await this.p2pNode.registerProtocol(BOB_API_PROTOCOL, apiRequestHandler);

        // Register result fetch handler
        const resultFetchHandler = createResultFetchHandler(this.queue);
        await this.p2pNode.registerProtocol(
            resultFetchHandler.protocol,
            resultFetchHandler.handler
        );

        console.log('P2P Provider Server started!');
        console.log(`Peer ID: ${this.p2pNode.getPeerId()}`);

        const multiaddrs = this.p2pNode.getMultiaddrs();
        console.log('Listening on multiaddrs:');
        multiaddrs.forEach(addr => console.log(`  ${addr}`));

        return this.p2pNode;
    }

    /**
     * Stop the P2P server
     */
    async stop() {
        if (this.p2pNode) {
            console.log('Stopping P2P Provider Server...');
            await this.p2pNode.stop();
            console.log('P2P Provider Server stopped');
        }
    }

    /**
     * Get the P2P node
     */
    getNode() {
        return this.p2pNode;
    }

    /**
     * Get multiaddrs for registration
     */
    getMultiaddrs() {
        if (!this.p2pNode) {
            return [];
        }
        return this.p2pNode.getMultiaddrs();
    }

    /**
     * Get public multiaddrs for registration
     */
    getPublicMultiaddrs() {
        if (!this.p2pNode) {
            return [];
        }
        return this.p2pNode.getPublicMultiaddrs();
    }
}

module.exports = P2PProviderServer;
