/**
 * Payment Verification Module
 *
 * Handles payment verification and replay protection
 */

class PaymentVerifier {
    constructor(database, solanaClient, config) {
        this.database = database;
        this.solana = solanaClient;
        this.config = config;
        this.providerWallet = config.wallet.address;
    }

    /**
     * Verify payment for API call
     *
     * @param {string} transactionSignature - Solana transaction signature
     * @param {number} expectedAmount - Expected payment amount
     * @param {object} queueCode - Queue code object with price and wallet
     * @returns {Promise<boolean>} - True if payment is valid
     */
    async verifyPayment(transactionSignature, expectedAmount, queueCode) {
        try {
            // Check if transaction already used (replay protection)
            if (await this.database.isTransactionUsed(transactionSignature)) {
                console.log(`Transaction already used: ${transactionSignature}`);
                return false;
            }

            // Verify on-chain
            const isValid = await this.solana.verifyPayment(
                transactionSignature,
                this.providerWallet,
                expectedAmount
            );

            if (!isValid) {
                console.log(`Payment verification failed for ${transactionSignature}`);
                return false;
            }

            // Additional verification: check if payment amount matches queue code price
            if (expectedAmount < queueCode.price) {
                console.log(`Payment amount (${expectedAmount}) less than queue price (${queueCode.price})`);
                return false;
            }

            console.log(`Payment verified: ${transactionSignature}`);
            return true;

        } catch (error) {
            console.error('Payment verification error:', error);
            return false;
        }
    }

    /**
     * Mark transaction as used
     *
     * @param {string} signature - Transaction signature
     * @param {string} jobId - Associated job ID
     * @param {number} amount - Payment amount
     */
    async markTransactionUsed(signature, jobId, amount) {
        await this.database.markTransactionUsed(signature, jobId, amount);
    }

    /**
     * Record earning for analytics
     *
     * @param {string} apiId - API identifier
     * @param {string} jobId - Job identifier
     * @param {number} amount - Amount earned
     * @param {string} walletAddress - Customer wallet address
     * @param {string} transactionSignature - Transaction signature
     */
    async recordEarning(apiId, jobId, amount, walletAddress, transactionSignature) {
        await this.database.recordEarning({
            apiId,
            jobId,
            amount,
            walletAddress,
            transactionSignature,
            earnedAt: Date.now()
        });
    }

    /**
     * Get earnings report
     */
    async getEarningsReport(startDate, endDate) {
        if (!startDate) {
            startDate = Date.now() - (30 * 24 * 60 * 60 * 1000); // 30 days ago
        }
        if (!endDate) {
            endDate = Date.now();
        }

        const earnings = await this.database.getEarnings(startDate, endDate);
        const total = await this.database.getTotalEarnings();

        // Group by API
        const byApi = {};
        for (const earning of earnings) {
            if (!byApi[earning.apiId]) {
                byApi[earning.apiId] = {
                    apiId: earning.apiId,
                    count: 0,
                    total: 0
                };
            }
            byApi[earning.apiId].count++;
            byApi[earning.apiId].total += earning.amount;
        }

        return {
            total,
            count: earnings.length,
            period: {
                start: startDate,
                end: endDate
            },
            byApi: Object.values(byApi),
            recent: earnings.slice(0, 10)
        };
    }
}

module.exports = PaymentVerifier;
